<?php

namespace App\Http\Controllers\Api\Student;

use App\Models\CartItem;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Validator;
use Stripe\Stripe;
use Stripe\PaymentIntent;

class CartController extends Controller
{
    public function addToCart(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'subject_id' => 'required|exists:subjects,id',
        ], [
            'subject_id.required' => 'المادة مطلوبة',
            'subject_id.exists' => 'المادة غير موجودة',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'message' => 'خطأ في البيانات المدخلة',
                'errors' => $validator->errors()
            ], 422);
        }

        $student = auth('student')->user();

        if (!$student) {
            return response()->json(['message' => 'الرجاء تسجيل الدخول'], 401);
        }

        $exists = CartItem::where('student_id', $student->id)
            ->where('subject_id', $request->subject_id)
            ->exists();

        if ($exists) {
            return response()->json(['message' => 'تمت إضافة المادة مسبقًا'], 409);
        }

        
        $student->cartItems()->create([
            'subject_id' => $request->subject_id
        ]);

        return response()->json(['message' => 'تمت إضافة المادة إلى السلة بنجاح']);
    }


    // card get all Item 
    public function getCart(Request $request)
    {
        $student = auth('student')->user();

        $cartItems = CartItem::with('subject')
            ->where('student_id', $student->id)
            ->get();

        // تصفية العناصر غير المدفوعة وجمع أسعارها
        $total = $cartItems->where('is_paid', false)
            ->sum(function ($item) {
                return $item->subject->price ?? 0;
            });

        // تجهيز البيانات
        $data = $cartItems->map(function ($item) {
            return [
                'id' => $item->id,
                'subject_id' => $item->subject_id,
                'subject_name' => $item->subject?->getTranslation('name', app()->getLocale()),
                'price' => $item->subject?->price,
                'is_paid' => $item->is_paid,
                'added_at' => $item->created_at->format('Y-m-d H:i')
            ];
        });

        return response()->json([
            'cart_items' => $data,
            'total_unpaid' => $total,
        ]);
    }

    // remove from cart 

    public function removeFromCart($lang, $id)
    {
        $student = auth('student')->user();

        $item = CartItem::where('student_id', $student->id)->where('id', $id)->first();

        if (!$item) {
            return response()->json([
                'message' => 'العنصر غير موجود في سلتك.',
            ], 404);
        }

        $item->delete();

        return response()->json([
            'message' => 'تم حذف العنصر من السلة بنجاح.',
            'item_id' => $id
        ], 200);
    }

    public function payCart(Request $request)
    {
        $student = auth('student')->user();

        $cartItems = CartItem::with('subject')
            ->where('student_id', $student->id)
            ->where('is_paid', false)
            ->get();

        if ($cartItems->isEmpty()) {
            return response()->json([
                'message' => 'السلة فارغة أو جميع المواد مدفوعة.',
            ], 400);
        }

        $totalAmount = $cartItems->sum(function ($item) {
            return $item->subject->price ?? 0;
        });

        // Stripe يعمل بالدولار أو عملة من نوع cent (1 دولار = 100 cent)
        $amountInCents = $totalAmount * 100;

        Stripe::setApiKey(config('services.stripe.secret'));

        $paymentIntent = PaymentIntent::create([
            'amount' => $amountInCents,
            'currency' => 'usd', // أو العملة المناسبة
            'metadata' => [
                'student_id' => $student->id,
                'cart_items' => $cartItems->pluck('id')->implode(','),
            ]
        ]);

        return response()->json([
            'client_secret' => $paymentIntent->client_secret,
            'amount' => $totalAmount,
            'currency' => 'usd',
        ]);
    }
}