<?php

namespace App\Http\Controllers\Api\Student;

use App\Models\Student;
use App\Models\Subject;
use App\Models\Classroom;
use Illuminate\Http\Request;
use Tymon\JWTAuth\Facades\JWTAuth;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;

class StudentAuthController extends Controller
{
    public function register(Request $request)
    {
        $lang = $request->query('lang', 'en');
        app()->setLocale($lang);

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:students,email',
            'password' => 'required|string|min:8|confirmed',
            'phone_number' => 'required|string|max:20',
            'parent_phone_number' => 'required|string|max:20',
            'country_id' => 'required|exists:countries,id',
            'state_id' => 'required|exists:states,id',
            'city_id' => 'required|exists:cities,id',
            'grade_id' => 'required|exists:grades,id',
            'classroom_id' => 'required|exists:classrooms,id',
            'term' => 'required|boolean',
            'heard_about_us' => 'nullable|string|max:255',
            'country_sign_in' => 'nullable|string|max:255',
            'university_card_image' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'device' => 'required|string',
            'device_type' => 'required|string',
            'ip_address' => 'required|string',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 400);
        }

        $classroom = Classroom::where('id', $request->classroom_id)
            ->where('grade_id', $request->grade_id)
            ->first();

        if (!$classroom) {
            return response()->json(['error' => 'الصف لا ينتمي إلى المرحلة المحددة'], 422);
        }

        $universityCardImageUrl = null;
        if ($request->hasFile('university_card_image')) {
            $file = $request->file('university_card_image');
            $filename = uniqid() . '.' . $file->getClientOriginalExtension();
            $file->move(public_path('university_card_image'), $filename);
            $universityCardImageUrl = url('university_card_image/' . $filename);
        }

        $student = Student::create([
            'name' => $request->name,
            'email' => $request->email,
            'phone_number' => $request->phone_number,
            'parent_phone_number' => $request->parent_phone_number,
            'country_id' => $request->country_id,
            'state_id' => $request->state_id,
            'city_id' => $request->city_id,
            'grade_id' => $request->grade_id,
            'heard_about_us_id' => $request->heard_about_us,
            'country_sign_in' => $request->country_sign_in,
            'university_card_image' => $universityCardImageUrl,
            'password' => bcrypt($request->password),
            'term' => $request->term,
            'device' => $request->device,
            'device_type' => $request->device_type,
            'ip_address' => $request->ip_address,
            'classroom_id' => $request->classroom_id,
        ]);

        $token = auth('student')->login($student);
        $student->update(['token' => $token]);

        return response()->json([
            'message' => __('messages.student_registered_successfully'),
            'token' => $token,
            'student' => $student,
        ], 201);
    }

    public function login(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'phone_number' => 'required',
            'password' => 'required|string|min:8',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()->all()], 400);
        }

        $lang = $request->query('lang', 'ar');

        $student = Student::where('phone_number', $request->phone_number)
            ->orWhere('parent_phone_number', $request->phone_number)
            ->first();

        if (!$student || !Hash::check($request->password, $student->password)) {
            return response()->json(['message' => 'Invalid credentials.'], 401);
        }

        $token = auth('student')->login($student);
        $student->update(['token' => $token]);

        $student->load('country', 'state', 'city', 'grade', 'classroom', 'heardAboutUs');

        return response()->json([
            'message' => 'Student login successful.',
            'token' => $token,
            'student' => [
                'id' => $student->id,
                'name' => $student->name,
                'email' => $student->email,
                'phone_number' => $student->phone_number,
                'parent_phone_number' => $student->parent_phone_number,
                'country' => [
                    'id' => $student->country?->id,
                    'name' => $student->country?->getTranslation('name', $lang),
                ],
                'state' => [
                    'id' => $student->state?->id,
                    'name' => $student->state?->getTranslation('name', $lang),
                ],
                'city' => [
                    'id' => $student->city?->id,
                    'name' => $student->city?->getTranslation('name', $lang),
                ],
                'grade' => [
                    'id' => $student->grade?->id,
                    'name' => $student->grade?->getTranslation('name', $lang),
                ],
                'classroom' => [
                    'id' => $student->classroom?->id,
                    'name' => $student->classroom?->getTranslation('name', $lang),
                ],
                'heard_about_us' => [
                    'id' => $student->heardAboutUs?->id,
                    'name' => $student->heardAboutUs?->getTranslation('name', $lang),
                ],
                'term' => $student->term,
                'university_card_image' => $student->university_card_image,
                'device' => $student->device,
                'device_type' => $student->device_type,
                'ip_address' => $student->ip_address,
            ],
        ], 200);
    }

    public function logout(Request $request)
    {
        try {
            auth('student')->logout();
            return response()->json(['message' => 'تم تسجيل الخروج بنجاح'], 200);
        } catch (\Exception $e) {
            return response()->json(['message' => 'فشل تسجيل الخروج'], 500);
        }
    }



    public function studentHome(Request $request)
    {
        $lang = app()->getLocale();

        $student = Student::with(['country', 'state', 'city', 'grade', 'heardAboutUs', 'classroom'])
            ->find(auth('student')->id());

        if (!$student) {
            return response()->json([
                'status' => false,
                'message' => 'الرجاء تسجيل الدخول أولاً',
            ], 401);
        }

        $studentData = [
            'id' => $student->id,
            'name' => $student->name,
            'email' => $student->email,
            'phone_number' => $student->phone_number,
            'parent_phone_number' => $student->parent_phone_number,
            'term' => $student->term,
            'country_sign_in' => $student->country_sign_in,
            'university_card_image' => $student->university_card_image,
            'device' => $student->device,
            'token' => $student->token,
            'ip_address' => $student->ip_address,
            'device_type' => $student->device_type,
            'country' => [
                'id' => $student->country?->id,
                'name' => $student->country?->getTranslation('name', $lang),
            ],
            'state' => [
                'id' => $student->state?->id,
                'name' => $student->state?->getTranslation('name', $lang),
            ],
            'city' => [
                'id' => $student->city?->id,
                'name' => $student->city?->getTranslation('name', $lang),
            ],
            'grade' => [
                'id' => $student->grade?->id,
                'name' => $student->grade?->getTranslation('name', $lang),
            ],
            'heard_about_us' => $student->heardAboutUs?->getTranslation('name', $lang),
        ];

        $subjects = Subject::where('grade_id', $student->grade_id)
            ->where('classroom_id', $student->classroom_id)
            ->get()
            ->map(function ($subject) use ($lang) {
                return [
                    'id' => $subject->id,
                    'name' => $subject->getTranslation('name', $lang),
                    'price' => $subject->price,
                ];
            });

        return response()->json([
            'status' => true,
            'message' => 'Student Home Data',
            'data' => [
                'student' => $studentData,
                'subjects' => $subjects,
            ]
        ]);
    }
}