<?php

namespace App\Http\Controllers;

use App\Models\BranchCategory;
use Illuminate\Http\Request;
use Yajra\DataTables\Facades\DataTables;

class BranchCategoryController extends Controller
{
    public function index(Request $request)
    {
        if ($request->ajax()) {
            $query = BranchCategory::select(['id', 'name', 'created_at']);

            return DataTables::of($query)
                ->addIndexColumn()
                ->addColumn('translated_name', function ($row) {
                    return $row->getTranslation('name', app()->getLocale());
                })
                ->addColumn('action', function ($row) {
                    $nameAr = e($row->getTranslation('name', 'ar'));
                    $nameEn = e($row->getTranslation('name', 'en'));

                    return '
                        <div class="d-flex justify-content-center align-items-center gap-2">
                            <button class="btn btn-primary btn-sm show-btn"
                                    data-id="' . $row->id . '"
                                    data-name_ar="' . $nameAr . '"
                                    data-name_en="' . $nameEn . '"
                                    aria-label="' . __('dashboard.show') . '"
                                    data-bs-toggle="tooltip" title="' . __('dashboard.show') . '">
                                <i class="fas fa-eye"></i>
                            </button>

                            <button class="btn btn-warning btn-sm edit-btn"
                                    data-id="' . $row->id . '"
                                    data-name_ar="' . $nameAr . '"
                                    data-name_en="' . $nameEn . '"
                                    aria-label="' . __('dashboard.edit') . '"
                                    data-bs-toggle="tooltip" title="' . __('dashboard.edit') . '">
                                <i class="fas fa-edit"></i>
                            </button>
                        </div>
                    ';
                })
                ->rawColumns(['action'])
                ->make(true);
        }

        return view('admin.pages.branch-categories.index');
    }

    public function store(Request $request)
    {
        $request->validate([
            'name.en' => 'required|string|max:255|unique:branch_categories,name->en',
            'name.ar' => 'required|string|max:255|unique:branch_categories,name->ar',
        ]);

        $category = BranchCategory::create([
            'name' => [
                'en' => $request->input('name.en'),
                'ar' => $request->input('name.ar'),
            ],
        ]);

        return response()->json([
            'success' => true,
            'message' => __('dashboard.created_successfully'),
            'data' => $category,
        ], 201);
    }

    public function show(BranchCategory $branch_category)
    {
        return response()->json([
            'id'   => $branch_category->id,
            'name' => $branch_category->getTranslations('name'),
        ]);
    }

    public function update(Request $request, BranchCategory $branch_category)
    {
        $request->validate([
            'name.en' => 'required|string|max:255|unique:branch_categories,name->en,' . $branch_category->id,
            'name.ar' => 'required|string|max:255|unique:branch_categories,name->ar,' . $branch_category->id,
        ]);

        $branch_category->update([
            'name' => [
                'en' => $request->input('name.en'),
                'ar' => $request->input('name.ar'),
            ],
        ]);

        return response()->json([
            'success' => true,
            'message' => __('dashboard.updated_successfully'),
        ]);
    }

    // public function destroy(BranchCategory $branch_category)
    // {
    //     $branch_category->delete();

    //     return back()->with('status', __('dashboard.deleted_successfully'));
    // }
}