<?php

namespace App\Http\Controllers;

use App\Models\City;
use App\Models\State;
use App\Models\Country;
use Illuminate\Http\Request;
use Yajra\DataTables\DataTables;

class CityController extends Controller
{
    public function index(Request $request)
    {
        if ($request->ajax()) {
            $cities = City::select([
                'cities.id',
                'cities.name',
                'cities.state_id',
                'states.id as state_id',
                'states.name as state_name',
                'countries.id as country_id',
                'countries.name as country_name', // Ensure correct alias
            ])
                ->leftJoin('states', 'cities.state_id', '=', 'states.id')
                ->leftJoin('countries', 'states.country_id', '=', 'countries.id'); // Join countries

            return DataTables::of($cities)
                ->filter(function ($query) use ($request) {
                    if ($search = $request->get('search')['value']) {
                        $query->where(function ($q) use ($search) {
                            // Search in both Arabic & English city, state, and country names
                            $q->where('cities.name->ar', 'LIKE', "%{$search}%")
                                ->orWhere('cities.name->en', 'LIKE', "%{$search}%")
                                ->orWhere('states.name->ar', 'LIKE', "%{$search}%")
                                ->orWhere('states.name->en', 'LIKE', "%{$search}%")
                                ->orWhere('countries.name->ar', 'LIKE', "%{$search}%")
                                ->orWhere('countries.name->en', 'LIKE', "%{$search}%");
                        });
                    }
                })
                ->addColumn('translated_name', function ($row) {
                    return $row->getTranslation('name', app()->getLocale());
                })
                ->addColumn('state_name', function ($row) {
                    return $row->state ? $row->state->getTranslation('name', app()->getLocale()) : '-';
                })
                ->addColumn('country_name', function ($row) {
                    return $row->state->country ? $row->state->country->getTranslation('name', app()->getLocale()) : '-';
                })
                ->addColumn('action', function ($row) {
                    return "<button
                    data-id='{$row->id}'
                    data-name_ar='{$row->getTranslation('name', 'ar')}'
                    data-name_en='{$row->getTranslation('name', 'en')}'
                    data-state_id='{$row->state_id}'
                    data-country_id='{$row->country_id}'
                    class='btn btn-danger edit-btn btn-sm'
                    data-bs-toggle='modal' data-bs-target='#editModal'>
                    <i class='fas fa-edit mx-2'></i> " .
                        __('dashboard.edit') .
                        "
                </button>";
                })
                ->orderColumn('translated_name', function ($query, $order) {
                    $query->orderBy('cities.name', $order);
                })
                ->orderColumn('state_name', function ($query, $order) {
                    $query->orderBy('states.name', $order);
                })
                ->orderColumn('country_name', function ($query, $order) {
                    $query->orderBy('countries.name', $order);
                })
                ->rawColumns(['action'])
                ->make(true);
        }

        $states = State::all();
        $countries = Country::get();
        return view('admin.pages.address.city.index', compact('states', 'countries'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name_ar' => 'required|string|max:255',
            'name_en' => 'required|string|max:255',
            'state_id' => 'required|exists:states,id', // Ensure state exists
        ]);

        City::create([
            'name' => ['ar' => $request->name_ar, 'en' => $request->name_en],
            'state_id' => $request->state_id,
        ]);

        toastr()->success(__('messages.success'));
        return redirect()->route('cities.index');
    }

    public function update(Request $request, $id)
    {
        $request->validate([
            'name' => 'required|array',
            'name.ar' => 'required|string',
            'name.en' => 'required|string',
            'state_id' => 'required|exists:states,id', // Corrected from `country_id`
        ]);

        $city = City::findOrFail($id);
        $city->setTranslation('name', 'ar', $request->name['ar']);
        $city->setTranslation('name', 'en', $request->name['en']);
        $city->state_id = $request->state_id; // Update state_id
        $city->save();

        return response()->json([
            'success' => true,
            'message' => __('messages.success'),
        ]);
    }

    public function CityState(Request $request)
    {
        // Validate input fields
        $validated = $request->validate([
            'name_ar' => 'required|string|max:255',
            'name_en' => 'required|string|max:255',
            'state_id' => 'required|exists:states,id',
        ]);

        // Create city
        City::create([
            'name' => ['ar' => $request->name_ar, 'en' => $request->name_en],
            'state_id' => $request->state_id,
        ]);

        // Show success message
        toastr()->success(__('messages.success'));
        return redirect()->route('cities.index');
    }
}
