<?php

namespace App\Http\Controllers;

use App\Models\Country;
use Illuminate\Http\Request;
use Yajra\DataTables\DataTables;

class CountryController extends Controller
{
    public function index(Request $request)
    {
        if ($request->ajax()) {
            $countries = Country::select(['id', 'name']);
            return DataTables::of($countries)
                ->filter(function ($query) use ($request) {
                    if ($search = $request->get('search')['value']) {
                        $query->where(function ($q) use ($search) {
                            $q->where('name->ar', 'LIKE', "%{$search}%")->orWhere('name->en', 'LIKE', "%{$search}%");
                        });
                    }
                })
                ->addColumn('translated_name', function ($row) {
                    return $row->getTranslation('name', app()->getLocale());
                })
                ->addColumn('action', function ($row) {
                    return "<button
                    data-id='{$row->id}'
                    data-name_ar='{$row->getTranslation('name', 'ar')}'
                    data-name_en='{$row->getTranslation('name', 'en')}'
                    class='btn  btn-danger edit-btn btn-sm' data-bs-toggle='modal' data-bs-target='#editModal'> <i
                                class='fas fa-edit mx-2'></i>
                    " .
                        __('dashboard.edit') .
                        "
                </button>";
                })
                ->rawColumns(['action'])
                ->make(true);
        }

        return view('admin.pages.address.country.index');
    }

    public function store(Request $request)
    {
        // Validate the Arabic and English name fields
        $validated = $request->validate([
            'name_ar' => 'required|string|max:255',
            'name_en' => 'required|string|max:255',
        ]);

        // Create a new entry in SchoolType with both Arabic and English names
        Country::create([
            'name' => ['ar' => $request->name_ar, 'en' => $request->name_en],
        ]);

        // Show success message and redirect
        toastr()->success(__('messages.success'));
        return redirect()->route('countries.index');
    }

    public function update(Request $request, $id)
    {
        $request->validate([
            'name' => 'required|array',
            'name.ar' => 'required|string',
            'name.en' => 'required|string',
        ]);

        $country = Country::findOrFail($id);
        $country->setTranslation('name', 'ar', $request->name['ar']);
        $country->setTranslation('name', 'en', $request->name['en']);
        $country->save();

        toastr()->success(__('messages.success'));

        return response()->json(['success' => true, 'message' => __('messages.success')]);
    }
}
