<?php

namespace App\Http\Controllers;

use App\Http\Requests\StoreCustomerRequest;
use App\Http\Requests\UpdateCustomerRequest;
use App\Models\Branch;
use App\Models\Customer;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Yajra\DataTables\Facades\DataTables;

class CustomerController extends Controller
{
    /** قائمة العملاء (DataTables + فلاتر) */
    public function index(Request $request)
    {
        $branches = Branch::orderBy('id')->get(['id', 'name', 'branch_code']);

        if ($request->ajax()) {
            $q = Customer::query()
                ->with(['branch:id,name,branch_code'])
                ->select(['id', 'customer_code', 'branch_id', 'name', 'created_at']);

            if ($request->filled('branch_id')) {
                $q->where('branch_id', (int) $request->branch_id);
            }
            if ($name = $request->string('name')->toString()) {
                $q->where('name', 'like', "%{$name}%");
            }

            return DataTables::of($q)
                ->addIndexColumn()
                ->addColumn('branch_name', function ($r) {
                    return method_exists($r->branch, 'getTranslation')
                        ? $r->branch->getTranslation('name', app()->getLocale())
                        : (is_array($r->branch->name) ? ($r->branch->name[app()->getLocale()] ?? ($r->branch->name['en'] ?? '')) : $r->branch->name);
                })
                ->addColumn('action', function ($r) {
                    $show = route('customers.show', $r->id);
                    $edit = route('customers.edit', $r->id);
                    return '
                        <div class="d-flex justify-content-center gap-1">
                          <a href="' . $show . '" class="btn btn-primary btn-sm" title="عرض"><i class="fas fa-eye"></i></a>
                          <a href="' . $edit . '" class="btn btn-warning btn-sm" title="تعديل"><i class="fas fa-edit"></i></a>
                        </div>';
                })
                ->rawColumns(['action'])
                ->make(true);
        }

        return view('admin.customers.index', compact('branches'));
    }

    /** شاشة الإضافة */
    public function create()
    {
        $user = auth()->user();
        $userBranchId = $user?->branch_id;
        $userBranch = null;

        if ($userBranchId) {
            $userBranch = Branch::find($userBranchId);
            $branches = collect([$userBranch]); // فرع واحد فقط
        } else {
            $branches = Branch::orderBy('id')->get(['id', 'name', 'branch_code']);
        }

        return view('admin.customers.form', compact('branches', 'userBranch', 'userBranchId'));
    }



    /** توليد كود متوقع (للواجهة – عند اختيار الفرع) */
    public function predictCode(Request $request)
    {
        $request->validate(['branch_id' => 'required|exists:branches,id']);
        $branch = Branch::findOrFail($request->branch_id);
        return response()->json([
            'code' => $this->predictNextCustomerCode($branch),
        ]);
    }

    /** حفظ جديد (كود نهائي داخل ترانزاكشن) */
    public function store(StoreCustomerRequest $request)
    {
        $branch = Branch::findOrFail($request->branch_id);

        $customer = null;

        DB::transaction(function () use ($request, $branch, &$customer) {

            // ✅ توليد كود جديد آمن
            $code = $this->generateNextCustomerCode($branch);

            // ✅ إنشاء العميل
            $customer = Customer::create([
                'customer_code' => $code,
                'branch_id'     => $branch->id,
                'name'          => $request->name,
            ]);
        });

        return redirect()
            ->route('customers.index')
            ->with('success', 'تمت إضافة العميل بنجاح');
    }


    /** عرض */
    public function show(Customer $customer)
    {
        $customer->load('branch');
        return view('admin.customers.show', compact('customer'));
    }

    /** شاشة التعديل */
    public function edit(Customer $customer)
    {
        $customer->load('branch');
        // لا نمكّن تغيير الفرع من الواجهة للمحافظة على منطق الكود
        return view('admin.customers.form', [
            'customer' => $customer,
            'branches' => Branch::orderBy('id')->get(['id', 'name', 'branch_code']),
        ]);
    }

    /** تحديث الاسم فقط */
    public function update(UpdateCustomerRequest $request, Customer $customer)
    {
        $customer->update([
            'name' => $request->name,
        ]);

        return redirect()->route('customers.index')
            ->with('success', 'تم تحديث بيانات العميل بنجاح');
    }

    // ================= Helpers =================

    /** بادئة الفرع = آخر رقمين من branch_code أو id عند غيابها */
    private function branchPrefix(Branch $branch): string
    {
        $code = $branch->branch_code ?: (string)$branch->id;
        // لو branch_code مثل "001" نأخذ آخر رقمين => "01"
        $num  = (int) $code;
        return str_pad((string)($num % 100), 2, '0', STR_PAD_LEFT);
    }

    /** توقع الكود التالي (عرض فقط) */
    protected function predictNextCustomerCode($branch)
    {
        // 🔹 احصل على آخر كود خاص بنفس الفرع
        $lastCode = \App\Models\Customer::where('branch_id', $branch->id)
            ->orderByDesc('id')
            ->value('customer_code');

        $lastNumber = 0;

        if ($lastCode && preg_match('/^\d{2}(\d{5})$/', $lastCode, $matches)) {
            $lastNumber = (int)$matches[1];
        }

        $nextNumber = $lastNumber + 1;

        // 🔹 البادئة = رقم الفرع بصيغة من رقمين
        $prefix = str_pad($branch->id, 2, '0', STR_PAD_LEFT);

        // 🔹 الجزء التسلسلي = رقم من 5 أرقام
        $sequence = str_pad($nextNumber, 5, '0', STR_PAD_LEFT);

        return $prefix . $sequence;
    }



    /** الكود النهائي (ترانزاكشن + lockForUpdate) */
    protected function generateNextCustomerCode($branch)
    {
        // 🔹 احصل على آخر كود خاص بنفس الفرع فقط
        $lastCode = Customer::where('branch_id', $branch->id)
            ->orderByDesc('id')
            ->value('customer_code');

        $lastNumber = 0;

        // 🔹 تحليل الكود واستخراج الرقم بعد أول رقمين (رمز الفرع)
        if ($lastCode && preg_match('/^\d{2}(\d{5})$/', $lastCode, $matches)) {
            $lastNumber = (int)$matches[1];
        }

        $nextNumber = $lastNumber + 1;

        // 🔹 رقم الفرع (مكون من رقمين)
        $prefix = str_pad($branch->id, 2, '0', STR_PAD_LEFT);

        // 🔹 الجزء التسلسلي (مكون من 5 أرقام)
        $sequence = str_pad($nextNumber, 5, '0', STR_PAD_LEFT);

        // ✅ النتيجة النهائية: YYxxxxx
        return $prefix . $sequence;
    }
}
