<?php
// app/Http/Controllers/MemberController.php

namespace App\Http\Controllers;

use App\Models\Member;
use App\Models\Branch;
use App\Models\Country;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rule;
use Carbon\Carbon;
use Yajra\DataTables\Facades\DataTables;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\File; // مهم
use Illuminate\Support\Str;

class MemberController extends Controller
{


    public function index(Request $request)
    {
        // للتصفية في الفلتر أعلى الصفحة
        $branches = \App\Models\Branch::orderBy('id')->get(['id', 'name', 'branch_code']);

        if ($request->ajax()) {
            try {
                $q = \App\Models\Member::query()
                    ->with(['branch:id,name,branch_code'])
                    ->select(['members.id', 'members.member_code', 'members.branch_id', 'members.status', 'members.created_at']);

                // فلاتر اختيارية
                if ($request->filled('branch_id')) {
                    $q->where('branch_id', (int) $request->branch_id);
                }
                if ($kw = $request->string('name')->toString()) {
                    // لو عندك حقل name نصّي عادي:
                    $q->where('name', 'like', "%{$kw}%");
                    // لو الاسم عندك Translatable JSON، غيّر للسطرين دول:
                    // $q->where('name->ar', 'like', "%{$kw}%")
                    //   ->orWhere('name->en', 'like', "%{$kw}%");
                }

                return DataTables::of($q)
                    ->addIndexColumn()
                    ->addColumn('branch_name', function ($r) {
                        $b = $r->branch;
                        if (!$b) return '—';
                        // لو name عندك translatable:
                        if (method_exists($b, 'getTranslation')) {
                            return $b->getTranslation('name', app()->getLocale());
                        }
                        // أو JSON:
                        if (is_array($b->name)) {
                            return $b->name[app()->getLocale()] ?? ($b->name['en'] ?? '');
                        }
                        // أو نص عادي:
                        return $b->name;
                    })
                    ->editColumn('created_at', fn($r) => optional($r->created_at)->format('Y-m-d'))
                    ->addColumn('status_badge', function ($r) {
                        return match ($r->status) {
                            'new'     => '<span class="badge bg-info">جديد</span>',
                            'active'  => '<span class="badge bg-success">نشط</span>',
                            default   => '<span class="badge bg-secondary">غير نشط</span>',
                        };
                    })
                    ->addColumn('action', function ($r) {
                        $show = route('members.show', $r->id);
                        $edit = route('members.edit', $r->id);
                        return '<div class="d-flex justify-content-center gap-1">
                        <a href="' . $show . '" class="btn btn-primary btn-sm" title="عرض"><i class="fas fa-eye"></i></a>
                        <a href="' . $edit . '" class="btn btn-warning btn-sm" title="تعديل"><i class="fas fa-edit"></i></a>
                    </div>';
                    })
                    ->rawColumns(['status_badge', 'action'])
                    ->toJson();
            } catch (\Throwable $e) {
                Log::error('DT members failed', ['msg' => $e->getMessage(), 'trace' => $e->getTraceAsString()]);
                return response()->json([
                    'error' => true,
                    'message' => $e->getMessage(),
                ], 500);
            }
        }

        return view('admin.members.index', compact('branches'));
    }


    /** شاشة إضافة */
    public function create()
    {
        return $this->formView(); // هيرجع الـ view ومعاه $branches و$countries
    }

    /** حفظ جديد */
    public function store(Request $request)
    {
        $this->validateStore($request);

        $branch = Branch::findOrFail($request->branch_id);

        $member = DB::transaction(function () use ($request, $branch) {

            // 1) كود العضو
            $code = $this->generateNextMemberCode($branch->id, $branch->branch_code, $branch->id);

            // 2) رفع الملفات
            // $paths = $this->handleUploads($request);

            $avatarPath       = $request->input('avatar_uploaded_path');        // من الرفع اللايف
            $idImagePath      = $request->input('id_image_uploaded_path');
            $workCardPath     = $request->input('work_card_uploaded_path');

            // fallback لو ما استُخدم الرفع اللايف
            if (!$avatarPath && $request->hasFile('avatar')) {
                $avatarPath = $request->file('avatar')->store('members/avatar', 'public');
            }
            if (!$idImagePath && $request->hasFile('id_image')) {
                $idImagePath = $request->file('id_image')->store('members/id', 'public');
            }
            if (!$workCardPath && $request->hasFile('work_card_image')) {
                $workCardPath = $request->file('work_card_image')->store('members/work-card', 'public');
            }

            // 3) العمر (إن وُجد ميلادي)
            $age = $request->filled('birth_greg')
                ? now()->diffInYears(Carbon::parse($request->birth_greg))
                : null;

            // 4) إنشاء
            return Member::create([
                'member_code' => $code,
                'branch_id'   => $branch->id,
                'status'      => $request->input('status', 'new'),

                'id_type'        => $request->id_type,
                'id_number'      => $request->id_number,
                'id_issue_greg'  => $request->id_issue_greg,
                'id_expiry_greg' => $request->id_expiry_greg,
                'id_issue_hijri' => $request->id_issue_hijri,
                'id_expiry_hijri' => $request->id_expiry_hijri,

                'name'           => $request->name,
                'gender'         => $request->gender,
                'nationality_id' => $request->nationality_id,

                'birth_greg'     => $request->birth_greg,
                'birth_hijri'    => $request->birth_hijri,
                'age'            => $age,

                'phone'          => $request->phone,
                'email'          => $request->email,
                'po_box'         => $request->po_box,
                'postal_code'    => $request->postal_code,
                'address'        => $request->address,

                'avatar_path'           => $request->input('avatar_uploaded_path'),
                'id_image_path'         => $request->input('id_image_uploaded_path'),
                'work_card_image_path'  => $request->input('work_card_image_uploaded_path'),

                'how_heard'      => $request->input('how_heard', []),
            ]);
        });

        return redirect()->route('members.show', $member)->with('success', 'تمت إضافة العضو بنجاح');
    }

    /** عرض */
    public function show(Member $member)
    {
        $member->load(['branch', 'nationality']);
        return view('admin.members.show', compact('member'));
    }

    /** تعديل */
    public function edit(Member $member)
    {
        return $this->formView($member);
    }
    /** تحديث */
    public function update(Request $request, Member $member)
    {
        $this->validateUpdate($request, $member);

        DB::transaction(function () use ($request, $member) {

            // $paths = $this->handleUploads($request, $member);

            $avatarPath       = $request->input('avatar_uploaded_path');        // من الرفع اللايف
            $idImagePath      = $request->input('id_image_uploaded_path');
            $workCardPath     = $request->input('work_card_uploaded_path');

            // fallback لو ما استُخدم الرفع اللايف
            if (!$avatarPath && $request->hasFile('avatar')) {
                $avatarPath = $request->file('avatar')->store('members/avatar', 'public');
            }
            if (!$idImagePath && $request->hasFile('id_image')) {
                $idImagePath = $request->file('id_image')->store('members/id', 'public');
            }
            if (!$workCardPath && $request->hasFile('work_card_image')) {
                $workCardPath = $request->file('work_card_image')->store('members/work-card', 'public');
            }

            $age = $request->filled('birth_greg')
                ? now()->diffInYears(Carbon::parse($request->birth_greg))
                : $member->age;

            $member->update([
                // member_code & branch_id لا تتغير عادةً
                'status'      => $request->input('status', $member->status),

                'id_type'        => $request->id_type,
                'id_number'      => $request->id_number,
                'id_issue_greg'  => $request->id_issue_greg,
                'id_expiry_greg' => $request->id_expiry_greg,
                'id_issue_hijri' => $request->id_issue_hijri,
                'id_expiry_hijri' => $request->id_expiry_hijri,

                'name'           => $request->name,
                'gender'         => $request->gender,
                'nationality_id' => $request->nationality_id,

                'birth_greg'     => $request->birth_greg,
                'birth_hijri'    => $request->birth_hijri,
                'age'            => $age,

                'phone'          => $request->phone,
                'email'          => $request->email,
                'po_box'         => $request->po_box,
                'postal_code'    => $request->postal_code,
                'address'        => $request->address,

                'avatar_path'           => $request->input('avatar_uploaded_path') ?: $member->avatar_path,
                'id_image_path'         => $request->input('id_image_uploaded_path') ?: $member->id_image_path,
                'work_card_image_path'  => $request->input('work_card_image_uploaded_path') ?: $member->work_card_image_path,

                'how_heard'      => $request->input('how_heard', []),
            ]);
        });

        return redirect()->route('members.show', $member)->with('success', 'تم تحديث بيانات العضو بنجاح');
    }

    /** حذف (اختياري) */
    public function destroy(Member $member)
    {
        $member->delete();
        return redirect()->route('members.index')->with('success', 'تم حذف العضو');
    }

    /** توقع الكود (للفورم) */
    public function predictCode(Request $request)
    {
        $request->validate(['branch_id' => ['required', 'exists:branches,id']]);

        $branch = Branch::select('id', 'branch_code')->findOrFail($request->branch_id);

        [$prefix, $next] = $this->peekSequence($branch->id, $branch->branch_code, $branch->id);
        return response()->json(['code' => $prefix . str_pad((string)$next, 5, '0', STR_PAD_LEFT)]);
    }

    private function formView(?Member $member = null)
    {
        $branches  = Branch::orderBy('id')->get(['id', 'name', 'branch_code']);
        $countries = Country::orderBy('id')->get(['id', 'name']);
        $isEdit    = (bool) $member;

        return view('admin.members.form', compact('member', 'branches', 'countries', 'isEdit'));
    }

    /* ===================== Helpers ===================== */

    /** قواعد التحقق أثناء الإضافة */
    private function validateStore(Request $request): void
    {
        $base = [
            'branch_id'      => ['required', 'exists:branches,id'],
            'status'         => ['nullable', Rule::in(['new', 'active', 'inactive'])],
            'id_type'        => ['nullable', Rule::in(['national_id', 'residence', 'passport', 'driving_license', 'work_card'])],
            'name'           => ['required', 'string', 'max:255'],
            'gender'         => ['nullable', Rule::in(['male', 'female'])],
            'nationality_id' => ['nullable', 'exists:countries,id'],
            'birth_greg'     => ['nullable', 'date'],
            'email'          => ['nullable', 'email'],
            'phone'          => ['nullable', 'string', 'max:30'],
            'po_box'         => ['nullable', 'string', 'max:50'],
            'postal_code'    => ['nullable', 'string', 'max:20'],
            'address'        => ['nullable', 'string', 'max:255'],
            'how_heard'      => ['nullable', 'array'],
            'avatar'         => ['nullable', 'image', 'max:2048'],
            'id_image'       => ['nullable', 'image', 'max:4096'],
            'work_card_image' => ['nullable', 'image', 'max:4096'],
        ];

        // شرط الهوية حسب النوع
        $rules = $base;
        if ($request->id_type === 'national_id') {
            $rules['id_number'] = ['required', 'regex:/^1\d{9}$/', 'unique:members,id_number'];
        } elseif ($request->id_type === 'residence') {
            $rules['id_number'] = ['required', 'regex:/^2\d{9}$/', 'unique:members,id_number'];
        } else {
            $rules['id_number'] = ['nullable', 'string', 'max:20', 'unique:members,id_number'];
        }

        $request->validate($rules);
    }

    /** قواعد التحقق أثناء التعديل */
    private function validateUpdate(Request $request, Member $member): void
    {
        $base = [
            'status'         => ['nullable', Rule::in(['new', 'active', 'inactive'])],
            'id_type'        => ['nullable', Rule::in(['national_id', 'residence', 'passport', 'driving_license', 'work_card'])],
            'name'           => ['required', 'string', 'max:255'],
            'gender'         => ['nullable', Rule::in(['male', 'female'])],
            'nationality_id' => ['nullable', 'exists:countries,id'],
            'birth_greg'     => ['nullable', 'date'],
            'email'          => ['nullable', 'email'],
            'phone'          => ['nullable', 'string', 'max:30'],
            'po_box'         => ['nullable', 'string', 'max:50'],
            'postal_code'    => ['nullable', 'string', 'max:20'],
            'address'        => ['nullable', 'string', 'max:255'],
            'how_heard'      => ['nullable', 'array'],
            'avatar'         => ['nullable', 'image', 'max:2048'],
            'id_image'       => ['nullable', 'image', 'max:4096'],
            'work_card_image' => ['nullable', 'image', 'max:4096'],
        ];

        $rules = $base;
        if ($request->id_type === 'national_id') {
            $rules['id_number'] = ['required', 'regex:/^1\d{9}$/', Rule::unique('members', 'id_number')->ignore($member->id)];
        } elseif ($request->id_type === 'residence') {
            $rules['id_number'] = ['required', 'regex:/^2\d{9}$/', Rule::unique('members', 'id_number')->ignore($member->id)];
        } else {
            $rules['id_number'] = ['nullable', 'string', 'max:20', Rule::unique('members', 'id_number')->ignore($member->id)];
        }

        $request->validate($rules);
    }

    /** رفع الملفات */
    private function handleUploads(Request $request, ?Member $existing = null): array
    {
        $disk = 'public';
        $out = [];

        if ($request->hasFile('avatar')) {
            if ($existing?->avatar_path) Storage::disk($disk)->delete($existing->avatar_path);
            $out['avatar'] = $request->file('avatar')->store('members/avatar', $disk);
        }
        if ($request->hasFile('id_image')) {
            if ($existing?->id_image_path) Storage::disk($disk)->delete($existing->id_image_path);
            $out['id_image'] = $request->file('id_image')->store('members/id', $disk);
        }
        if ($request->hasFile('work_card_image')) {
            if ($existing?->work_card_image_path) Storage::disk($disk)->delete($existing->work_card_image_path);
            $out['work_card'] = $request->file('work_card_image')->store('members/work-card', $disk);
        }
        return $out;
    }

    /** اعادة بناء بادئة الكود من الفرع */
    private function branchPrefix(?string $branchCode, int $branchId): string
    {
        $base = $branchCode ?: (string)$branchId;
        return str_pad(substr($base, -2), 2, '0', STR_PAD_LEFT);
    }

    /**
     * قراءة التسلسل المتوقع (بدون زيادة) — لعرضه في الفورم
     * تُعيد [prefix, nextNumber]
     */
    private function peekSequence(int $branchId, ?string $branchCode, int $fallbackId): array
    {
        $prefix = $this->branchPrefix($branchCode, $fallbackId);

        $row = DB::table('member_sequences')->where('branch_id', $branchId)->first();
        if (!$row) return [$prefix, 1];

        $next = (int)($row->last_seq ?? 0) + 1;
        return [$row->prefix ?: $prefix, $next];
    }

    /**
     * يولّد كود جديد آمن (داخل ترانزاكشن)
     */
    private function generateNextMemberCode(int $branchId, ?string $branchCode, int $fallbackId): string
    {
        return DB::transaction(function () use ($branchId, $branchCode, $fallbackId) {
            $prefix = $this->branchPrefix($branchCode, $fallbackId);

            // اقفل الصف المعنيّ
            $row = DB::table('member_sequences')
                ->lockForUpdate()
                ->where('branch_id', $branchId)
                ->first();

            if (!$row) {
                DB::table('member_sequences')->insert([
                    'branch_id' => $branchId,
                    'prefix'    => $prefix,
                    'last_seq'  => 0,
                    'created_at' => now(),
                    'updated_at' => now(),
                ]);
                $row = (object)['branch_id' => $branchId, 'prefix' => $prefix, 'last_seq' => 0];
            }

            $newSeq = ((int)$row->last_seq) + 1;

            DB::table('member_sequences')
                ->where('branch_id', $branchId)
                ->update(['last_seq' => $newSeq, 'updated_at' => now()]);

            $finalPrefix = $row->prefix ?: $prefix;

            return $finalPrefix . str_pad((string)$newSeq, 5, '0', STR_PAD_LEFT);
        });
    }


    public function upload(Request $request)
    {
        // field = avatar | id_image | work_card_image
        $request->validate([
            'file'  => 'required|image|mimes:jpg,jpeg,png,webp|max:5120',
            'field' => 'required|in:avatar,id_image,work_card_image',
        ]);

        $map = [
            'avatar'          => 'uploads/members/avatar',
            'id_image'        => 'uploads/members/id',
            'work_card_image' => 'uploads/members/work-card',
        ];

        $subdir = $map[$request->field];
        $dest   = public_path($subdir);

        if (! File::exists($dest)) {
            File::makeDirectory($dest, 0775, true);
        }

        $ext = $request->file('file')->getClientOriginalExtension();
        $name = date('Ymd_His') . '_' . Str::random(8) . '.' . $ext;

        // move إلى public/
        $request->file('file')->move($dest, $name);

        $relative = $subdir . '/' . $name;    // مثال: uploads/members/avatar/xxxx.jpg
        $url      = asset($relative);

        return response()->json([
            'ok'   => true,
            'path' => $relative, // خزنها في DB لاحقًا
            'url'  => $url,      // للمعاينة
        ]);
    }

    public function deleteUpload(Request $request)
    {
        $request->validate([
            'path' => 'required|string'
        ]);

        $path = $request->path;

        // أمان: امنع المسارات خارج uploads/members/
        if (!Str::startsWith($path, 'uploads/members/')) {
            return response()->json(['ok' => false, 'msg' => 'Invalid path'], 422);
        }

        $full = public_path($path);
        if (File::exists($full)) {
            File::delete($full);
        }

        return response()->json(['ok' => true]);
    }
}
