<?php

namespace App\Http\Controllers;

use App\Http\Requests\StoreOfferRequest;
use App\Http\Requests\UpdateOfferRequest;
use App\Models\Offer;
use App\Models\OfferItem;
use App\Models\Service;
use App\Models\Member;
use App\Models\Customer;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Yajra\DataTables\Facades\DataTables;

class OfferController extends Controller
{

    public function index(Request $request)
    {
        if ($request->ajax()) {
            $q = Offer::query()
                ->withCount('items') // items_count
                ->select([
                    'id',
                    'offer_no',
                    'name',
                    'scope',
                    'start_date_g',
                    'end_date_g',
                    'duration_days',
                    'discount_type',
                    'discount_value',
                    'extra_days',
                    'active',
                    'created_at'
                ]);

            // بحث بالاسم (يقبل search_name أو name)
            $search = $request->string('search_name')->toString()
                ?: $request->string('name')->toString();
            if ($search !== '') {
                $q->where(function ($qq) use ($search) {
                    $qq->where('name->ar', 'like', "%{$search}%")
                        ->orWhere('name->en', 'like', "%{$search}%");
                });
            }

            // فلتر النطاق (تجاهل القيمة الفارغة)
            $scope = $request->string('scope')->toString();
            if ($scope !== '' && in_array($scope, ['all', 'members', 'customers', 'coupons'], true)) {
                $q->where('scope', $scope);
            }

            // الحالة
            if ($request->filled('active')) {
                $q->where('active', (int)$request->active === 1);
            }

            // المدى التاريخي (اختياري): نفلتر بالعروض اللي تقع تمامًا داخل المدى
            $from = $request->date('from');
            $to   = $request->date('to');
            if ($from && $to) {
                $q->whereDate('start_date_g', '>=', $from)
                    ->whereDate('end_date_g',   '<=', $to);
            } elseif ($from) {
                $q->whereDate('start_date_g', '>=', $from);
            } elseif ($to) {
                $q->whereDate('end_date_g', '<=', $to);
            }

            return \Yajra\DataTables\Facades\DataTables::of($q)
                ->addIndexColumn()
                ->addColumn('translated_name', function ($r) {
                    if (is_array($r->name)) {
                        $loc = app()->getLocale();
                        return $r->name[$loc] ?? ($r->name['en'] ?? reset($r->name));
                    }
                    return $r->name;
                })
                ->addColumn('scope_badge', function ($r) {
                    $map = [
                        'all'       => 'جميع الأعضاء/العملاء',
                        'members'   => 'الأعضاء',
                        'customers' => 'العملاء',
                        'coupons'   => 'كوبونات',
                    ];
                    return '<span class="badge bg-info">' . ($map[$r->scope] ?? e($r->scope)) . '</span>';
                })
                ->addColumn('date_range', function ($r) {
                    $start = $r->start_date_g ? \Carbon\Carbon::parse($r->start_date_g)->format('Y-m-d') : '';
                    $end   = $r->end_date_g   ? \Carbon\Carbon::parse($r->end_date_g)->format('Y-m-d')   : '';
                    return e($start) . ' → ' . e($end);
                })

                ->addColumn('discount_summary', function ($r) {
                    return $r->discount_type === 'percent'
                        ? number_format((float)$r->discount_value, 2) . ' %'
                        : number_format((float)$r->discount_value, 2);
                })

                ->addColumn(
                    'status_badge',
                    fn($r) =>
                    $r->active
                        ? '<span class="badge bg-success">نشط</span>'
                        : '<span class="badge bg-secondary">غير نشط</span>'
                )
                ->addColumn('action', function ($r) {
                    $show = route('offers.show', $r->id);
                    $edit = route('offers.edit', $r->id);
                    $toggle = route('offers.toggle-active', $r->id);

                    $toggleLabel = $r->active ? 'إلغاء تنشيط' : 'تنشيط';
                    $toggleClass = $r->active ? 'btn-danger' : 'btn-success';
                    $icon = $r->active ? 'fa-ban' : 'fa-check';

                    return '
        <div class="d-flex justify-content-center gap-1">
            <a href="' . $show . '" class="btn btn-primary btn-sm" title="عرض">
                <i class="fas fa-eye"></i>
            </a>
            <a href="' . $edit . '" class="btn btn-warning btn-sm" title="تعديل">
                <i class="fas fa-edit"></i>
            </a>
            <a href="javascript:void(0);" 
               data-url="' . $toggle . '" 
               data-id="' . $r->id . '" 
               class="btn ' . $toggleClass . ' btn-sm btnToggleActive" 
               title="' . $toggleLabel . '">
               <i class="fas ' . $icon . '"></i>
            </a>
        </div>';
                })

                ->rawColumns(['scope_badge', 'status_badge', 'action'])
                ->make(true);
        }

        return view('admin.offers.index');
    }

    public function toggleActive(Offer $offer)
    {
        $offer->active = !$offer->active;
        $offer->save();

        // تحقق أنه فعلاً تغيّر في قاعدة البيانات
        $offer->refresh();

        return response()->json([
            'success' => true,
            'new_status' => $offer->active,
            'message' => $offer->active
                ? '✅ تم تنشيط العرض بنجاح.'
                : '🚫 تم إلغاء تنشيط العرض بنجاح.'
        ]);
    }




    /** إعطاء رقم العرض التالي داخل ترانزاكشن */
    private function allocateOfferNo(): int
    {
        $max = (int) DB::table('offers')->lockForUpdate()->max('offer_no');
        return $max + 1;
    }

    public function show(Offer $offer)
    {
        $offer->load('items.service'); // مهم لقراءة أسماء الخدمات الحية
        return view('admin.offers.show', compact('offer'));
    }

    public function create()
    {
        $nextNo    = (int) DB::table('offers')->max('offer_no') + 1;
        $services  = Service::orderBy('id', 'desc')->get(['id', 'name', 'price', 'duration_value', 'duration_unit']);
        $members   = Member::orderBy('id', 'desc')->get(['id', 'name', 'member_code']);
        $customers = Customer::orderBy('id', 'desc')->get(['id', 'name', 'customer_code']);

        return view('admin.offers.create', compact('services', 'members', 'customers', 'nextNo'));
    }

    public function store(StoreOfferRequest $request)
    {

        // dd($request);
        $data = $request->validated();

        // تأكيد وجود عناصر (إضافي بجانب الـ Request)
        if (empty($data['items']) || count($data['items']) < 1) {
            return back()->withErrors(['items' => 'يجب إضافة عنصر واحد على الأقل.'])->withInput();
        }

        // فض الاشتباك في النطاق
        $scope = $data['scope'];
        $targetMembers   = $scope === 'members'   ? ($data['target_member_ids'] ?? null)   : null;
        $targetCustomers = $scope === 'customers' ? ($data['target_customer_ids'] ?? null) : null;
        $couponCodes     = $scope === 'coupons'   ? ($data['coupon_codes'] ?? null)        : null;

        // حساب المدة
        $durationDays = \Carbon\Carbon::parse($data['end_date_g'])
            ->diffInDays(\Carbon\Carbon::parse($data['start_date_g'])) + 1;

        DB::transaction(function () use ($data, $scope, $targetMembers, $targetCustomers, $couponCodes, $durationDays) {
            $offerNo = $this->allocateOfferNo();

            $offer = Offer::create([
                'offer_no'          => $offerNo,
                'name'              => $data['name'],
                'scope'             => $scope,
                'target_member_ids' => $targetMembers,
                'target_customer_ids' => $targetCustomers,
                'coupon_codes'      => $couponCodes,
                'start_date_g'      => $data['start_date_g'],
                'end_date_g'        => $data['end_date_g'],
                'start_date_h'      => $data['start_date_h'] ?? null,
                'end_date_h'        => $data['end_date_h'] ?? null,
                'duration_days'     => $durationDays,
                'discount_type'     => $data['discount_type'],
                'discount_value'    => $data['discount_value'],
                'extra_days'        => $data['extra_days'] ?? 0,
                'active'            => (bool)($data['active'] ?? true),
            ]);

            $this->persistItems(
                offer: $offer,
                items: $data['items'],
                gType: $offer->discount_type,
                gVal: (float)$offer->discount_value,
                gExtra: (int)$offer->extra_days
            );
        });

        return redirect()->route('offers.index')->with('success', 'تم إنشاء العرض بكل عناصره بنجاح.');
    }

    public function edit(Offer $offer)
    {
        $offer->load(['items.service']);
        $services  = Service::orderBy('id', 'desc')->get(['id', 'name', 'price', 'duration_value', 'duration_unit']);
        $members   = Member::orderBy('id', 'desc')->get(['id', 'name', 'member_code']);
        $customers = Customer::orderBy('id', 'desc')->get(['id', 'name', 'customer_code']);
        return view('admin.offers.edit', compact('offer', 'services', 'members', 'customers'));
    }

    public function update(UpdateOfferRequest $request, Offer $offer)
    {
        $data = $request->validated();

        if (empty($data['items']) || count($data['items']) < 1) {
            return back()->withErrors(['items' => 'لا يمكن حفظ العرض بدون خدمات.'])->withInput();
        }

        $scope = $data['scope'];
        $targetMembers   = $scope === 'members'   ? ($data['target_member_ids'] ?? null)   : null;
        $targetCustomers = $scope === 'customers' ? ($data['target_customer_ids'] ?? null) : null;
        $couponCodes     = $scope === 'coupons'   ? ($data['coupon_codes'] ?? null)        : null;

        $durationDays = \Carbon\Carbon::parse($data['end_date_g'])
            ->diffInDays(\Carbon\Carbon::parse($data['start_date_g'])) + 1;

        DB::transaction(function () use ($offer, $data, $scope, $targetMembers, $targetCustomers, $couponCodes, $durationDays) {
            $offer->update([
                'name'              => $data['name'],
                'scope'             => $scope,
                'target_member_ids'   => $targetMembers,
                'target_customer_ids' => $targetCustomers,
                'coupon_codes'        => $couponCodes,
                'start_date_g'        => $data['start_date_g'],
                'end_date_g'          => $data['end_date_g'],
                'start_date_h'        => $data['start_date_h'] ?? null,
                'end_date_h'          => $data['end_date_h'] ?? null,
                'duration_days'       => $durationDays,
                'discount_type'       => $data['discount_type'],
                'discount_value'      => $data['discount_value'],
                'extra_days'          => $data['extra_days'] ?? 0,
                'active'              => (bool)($data['active'] ?? $offer->active),
            ]);

            // امسح القديم، ثم اكتب العناصر الجديدة (تضمن التطابق مع الشاشة)
            $offer->items()->delete();
            $this->persistItems(
                offer: $offer,
                items: $data['items'],
                gType: $offer->discount_type,
                gVal: (float)$offer->discount_value,
                gExtra: (int)$offer->extra_days
            );
        });

        return redirect()->route('offers.index')->with('success', 'تم تحديث العرض وعناصره بنجاح.');
    }

    /** Endpoint لجلب تفاصيل خدمة واحدة (يُستخدم في الواجهة) */
    public function serviceInfo(Service $service)
    {
        return response()->json([
            'id'             => $service->id,
            'name'           => $service->getTranslations('name'),
            'price'          => (float) $service->price,
            'duration_value' => (int) $service->duration_value,
            'duration_unit'  => $service->duration_unit,
        ]);
    }

    /** كتابة عناصر العرض مع احتساب النهائي ومنع التكرار */
    private function persistItems(Offer $offer, array $items, string $gType, float $gVal, int $gExtra): void
    {
        $seen = [];

        foreach ($items as $row) {
            $sid = (int)($row['service_id'] ?? 0);
            if ($sid <= 0) continue;
            if (isset($seen[$sid])) continue; // منع التكرار بالمنطق
            $seen[$sid] = true;

            $svc = Service::findOrFail($sid);

            $basePrice   = (float)$svc->price;
            $baseDurVal  = (int)$svc->duration_value;
            $baseDurUnit = $svc->duration_unit;

            $dtype = $row['discount_type'] ?? $gType;
            $dval  = array_key_exists('discount_value', $row) && $row['discount_value'] !== ''
                ? (float)$row['discount_value'] : (float)$gVal;
            $xDays = array_key_exists('extra_days', $row) && $row['extra_days'] !== ''
                ? (int)$row['extra_days'] : (int)$gExtra;

            $finalPrice = $dtype === 'percent'
                ? max(0, $basePrice - ($basePrice * ($dval / 100)))
                : max(0, $basePrice - $dval);

            // نزيد الأيام فقط لو وحدة المدة أيام
            $finalDurVal  = $baseDurUnit === 'days' ? $baseDurVal + $xDays : $baseDurVal;

            OfferItem::create([
                'offer_id' => $offer->id,
                'service_id' => $svc->id,

                'service_name' => $svc->name, // JSON {ar,en}
                'base_price'   => $basePrice,
                'base_duration_value' => $baseDurVal,
                'base_duration_unit'  => $baseDurUnit,

                // القيم الخاصة بالصف إن أردت تفعيلها مستقبلاً
                'row_discount_type'  => $row['discount_type'] ?? null,
                'row_discount_value' => array_key_exists('discount_value', $row) ? ($row['discount_value'] ?? null) : null,
                'row_extra_days'     => array_key_exists('extra_days', $row) ? ($row['extra_days'] ?? null) : null,

                'final_price'          => $finalPrice,
                'final_duration_value' => $finalDurVal,
                'final_duration_unit'  => $baseDurUnit,
            ]);
        }
    }
}