<?php

namespace App\Http\Controllers;

use App\Models\Subject;
use Illuminate\Http\Request;
use Stripe\Stripe;
use Stripe\Checkout\Session;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class PaymentController extends Controller
{
    public function createStripeSession(Request $request)
{
    $student = auth('student')->user();
    $lang = app()->getLocale();

    $request->validate([
        'subject_id' => 'required|exists:subjects,id',
    ]);

    $subject = Subject::findOrFail($request->subject_id);

    Stripe::setApiKey(env('STRIPE_SECRET'));

    // Create checkout session
    $session = Session::create([
        'payment_method_types' => ['card'],
        'line_items' => [[
            'price_data' => [
                'currency' => 'usd',
                'product_data' => [
                    'name' => $subject->getTranslation('name', $lang),
                ],
                'unit_amount' => $subject->price * 100, // Stripe uses cents
            ],
            'quantity' => 1,
        ]],
        'mode' => 'payment',
        'success_url' => route('payment.success') . '?session_id={CHECKOUT_SESSION_ID}',
        'cancel_url' => route('payment.cancel'),
        'metadata' => [
            'student_id' => $student->id,
            'subject_id' => $subject->id,
        ],
    ]);

    return response()->json(['url' => $session->url]);
}

public function paymentSuccess(Request $request)
{
    Stripe::setApiKey(env('STRIPE_SECRET'));

    $sessionId = $request->get('session_id');

    if (!$sessionId) {
        return response()->json(['error' => 'Missing session ID'], 400);
    }

    $session = \Stripe\Checkout\Session::retrieve($sessionId);

    $studentId = $session->metadata->student_id ?? null;
    $subjectId = $session->metadata->subject_id ?? null;

    if (!$studentId || !$subjectId) {
        return response()->json(['error' => 'Metadata missing'], 400);
    }

    // Save payment record
    DB::table('payments')->insert([
        'student_id' => $studentId,
        'subject_id' => $subjectId,
        'amount' => $session->amount_total / 100,
        'payment_date' => now(),
        'expires_at' => Carbon::now()->addMonths(3), // مثلاً: تنتهي بعد 3 أشهر
        'created_at' => now(),
        'updated_at' => now(),
    ]);

    return response()->json([
        'status' => true,
        'message' => 'تم الدفع بنجاح',
    ]);
}

}
