<?php

namespace App\Http\Controllers;

use App\Models\Exam;
use App\Models\Question;
use Illuminate\Http\Request;
use App\Models\QuestionGroup;
use App\Models\QuestionChoice;
use App\Models\QuestionEssayAnswer;

class QuestionController extends Controller
{
    public function question()
    {
        $exams = Exam::get();
        $groups = QuestionGroup::get();
        return view('admin.pages.subject.addExam', compact('exams', 'groups'));
    }


    public function store(Request $request)
    {

      // return $request;
        // Validate Request
        $validatedData = $request->validate([
            'exams' => 'required|exists:exams,id',
            'groups' => 'required|exists:question_groups,id',
            'question_text_ar' => 'required|string|max:255',
            'question_text_en' => 'required|string|max:255',
            'marks' => 'required|integer|min:1',
            'question_type' => 'required|integer|in:1,2',

            // Video Path Validation (Uploaded in Background)
            'uploaded_video' => 'nullable|string',

            // Multiple Choice Validation (if question type is 1)
            'choices' => 'nullable|array|required_if:question_type,1',
            'choices.*.name_ar' => 'nullable|required_if:question_type,1|string|max:255',
            'choices.*.name_en' => 'nullable|required_if:question_type,1|string|max:255',
            'choices.*.correct' => 'nullable|sometimes|in:0,1', // Ensure correct value is 0 or 1

          

            // Essay Validation (Only when question type is 2)
            'essay' => 'nullable|array|required_if:question_type,2|min:1',
            'essay.*.answer_ar' => 'nullable|required_if:question_type,2|string|max:255',
            'essay.*.answer_en' => 'nullable|required_if:question_type,2|string|max:255',
            'essay.*.correct_answer' => 'nullable|required_if:question_type,2|string|max:255',
        ]);

        // return $request;

        // Handle Video Upload
        $videoPath = $request->uploaded_video ?? null;

        // Create the Question
        $question = Question::create([
            'exam_id' => $request->exams,
            'group_id' => $request->groups,
            'question_text' => [
                'ar' => $request->question_text_ar,
                'en' => $request->question_text_en,
            ],
            'marks' => $request->marks,
            'question_type' => $request->question_type,
            'video' => $videoPath,
            'is_resolve' => $request->is_resolve ?? 0,
        ]);

        // Handle Multiple Choice
        if ($request->question_type == 1 && $request->has('choices')) {
            foreach ($request->choices as $choice) {

              
                QuestionChoice::create([
                    'question_id' => $question->id,
                    'choice_text' => [
                        'ar' => $choice['name_ar'],
                        'en' => $choice['name_en'],
                    ],
                    'is_correct' => isset($choice['correct']) && $choice['correct'] == '1',
                ]);
            }
        }

        // Handle Essay Answers
        if ($request->question_type == 2 && $request->has('essay')) {
            foreach ($request->essay as $answer) {
                QuestionEssayAnswer::create([
                    'question_id' => $question->id,
                    'user_answer' => [
                        'ar' => $answer['answer_ar'],
                        'en' => $answer['answer_en'],
                    ],
                    'correct_answer' =>$answer['correct_answer'] ,
                ]);
            }
        }

        toastr()->success(__('messages.success'));
        return redirect()->back();
    }

    public function uploadVideo(Request $request)
    {
        $request->validate([
            'video' => 'required|file|mimes:mp4,mov,avi,mkv|max:51200', // Allow up to 50MB
        ]);

        if ($request->hasFile('video')) {
            $path = $request->file('video')->store('videos', 'public'); // Save to storage/public/videos
            return response()->json(['filepath' => $path]);
        }

        return response()->json(['error' => 'File upload failed'], 400);
    }


    public function show($id)
    {
       // جلب الامتحان مع الأسئلة والمجموعات
       $exam = Exam::with(['questions.group', 'questions.choices', 'questions.essayAnswers'])
       ->findOrFail($id);

// تجميع الأسئلة حسب المجموعة
$groupedQuestions = $exam->questions->groupBy('group_id');

return view('admin.pages.questions.show', compact('exam', 'groupedQuestions'));
        
      
    }

}
