<?php

namespace App\Http\Controllers;

use App\Models\State;
use App\Models\Country;
use Illuminate\Http\Request;
use Yajra\DataTables\DataTables;

class StateController extends Controller
{
  public function index(Request $request)
  {
      if ($request->ajax()) {
          $states = State::select([
                  'states.id',
                  'states.name',
                  'states.country_id',
                  'countries.name as country_name', // Correct alias
              ])
              ->leftJoin('countries', 'states.country_id', '=', 'countries.id'); // Ensure join is correct
  
          return DataTables::of($states)
              ->filter(function ($query) use ($request) {
                  if ($search = $request->get('search')['value']) {
                      $query->where(function ($q) use ($search) {
                          // Search in both Arabic & English names of the states
                          $q->where('states.name->ar', 'LIKE', "%{$search}%")
                              ->orWhere('states.name->en', 'LIKE', "%{$search}%")
                              // Also search in both Arabic & English names of the country
                              ->orWhereHas('country', function ($q) use ($search) {
                                  $q->where('countries.name->ar', 'LIKE', "%{$search}%")
                                      ->orWhere('countries.name->en', 'LIKE', "%{$search}%");
                              });
                      });
                  }
              })
              ->addColumn('translated_name', function ($row) {
                  return $row->getTranslation('name', app()->getLocale()); // Return translated name based on locale
              })
              ->addColumn('country_name', function ($row) {
                  return $row->country ? $row->country->getTranslation('name', app()->getLocale()) : '-'; // Return translated country name
              })
              ->addColumn('action', function ($row) {
                  return "<button
                      data-id='{$row->id}'
                      data-name_ar='{$row->getTranslation('name', 'ar')}'
                      data-name_en='{$row->getTranslation('name', 'en')}'
                      data-country_id='{$row->country_id}'
                      class='btn btn-danger edit-btn btn-sm'
                      data-bs-toggle='modal' data-bs-target='#editModal'>
                      <i class='fas fa-edit mx-2'></i> " . __('dashboard.edit') . "
                  </button>";
              })
              ->orderColumn('translated_name', function ($query, $order) {
                  $query->orderBy('states.name', $order);
              })
              ->orderColumn('country_name', function ($query, $order) {
                  $query->orderBy('countries.name', $order);
              })
              ->rawColumns(['action']) // Make sure the 'action' column is rendered as raw HTML
              ->make(true);
      }
  
      $countries = Country::get();
      return view('admin.pages.address.state.index', compact('countries'));
  }
  

    public function store(Request $request)
    {
        // Validate the Arabic and English name fields
        $validated = $request->validate([
            'name_ar' => 'required|string|max:255',
            'name_en' => 'required|string|max:255',
            'country_id' => 'required',
        ]);

        // Create a new entry in SchoolType with both Arabic and English names
        State::create([
            'name' => ['ar' => $request->name_ar, 'en' => $request->name_en],
            'country_id' => $request->country_id,
        ]);

        // Show success message and redirect
        toastr()->success(__('messages.success'));
        return redirect()->route('states.index');
    }

    public function update(Request $request, $id)
    {
        $request->validate([
            'name' => 'required|array',
            'name.ar' => 'required|string',
            'name.en' => 'required|string',
            'country_id' => 'required|exists:countries,id',
        ]);

        $state = State::findOrFail($id);
        $state->setTranslation('name', 'ar', $request->name['ar']);
        $state->setTranslation('name', 'en', $request->name['en']);
        $state->country_id = $request->country_id; // Update country_id
        $state->save();

        return response()->json([
            'success' => true,
            'message' => __('messages.success'),
        ]);
    }
}
