<?php

namespace App\Http\Controllers;

use App\Models\Role;
use App\Models\User;
use App\Models\Branch;
use Illuminate\Http\Request;
use Yajra\DataTables\DataTables;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;

class UserController extends Controller
{
    // Auth Functions **************************
    public function loginForm()
    {
        return view('auth.login');
    }
    public function login(Request $request)
    {
        $request->validate([
            'email' => 'required|email',
            'password' => 'required|min:6',
        ]);

        $credentials = $request->only('email', 'password');

        if (Auth::guard('web')->attempt($credentials)) {
            toastr()->success(__('messages.welcome_admin'));
            return redirect()->route('admin.dashboard');
        }

        toastr()->error(__('messages.invalid_credentials'));
        return back()
            ->withErrors(['error' => __('messages.invalid_credentials')])
            ->withInput();
    }

    public function logout(Request $request)
    {
        // تسجيل الخروج من الجارد الحالي
        Auth::guard('web')->logout();

        // إلغاء الجلسة تمامًا
        $request->session()->invalidate();
        $request->session()->regenerateToken();

        // رسالة نجاح
        toastr()->success(__('messages.logged_out'));

        // إعادة التوجيه للصفحة الرئيسية أو صفحة الدخول
        return redirect()->route('admin.login');
    }


    // End Auth Functions **********************

    // users Managment ************************

    public function index()
    {
        $roles = Role::all();
        $branches = Branch::all();
        return view('users.index', compact('roles', 'branches'));
    }

    public function list(Request $request)
    {
        $query = User::with('roles', 'branch');

        if ($request->name) {
            $query->where('name', 'like', "%{$request->name}%");
        }

        if ($request->branch_id) {
            $query->where('branch_id', $request->branch_id);
        }

        $users = $query->get();

        return DataTables::of($users)
            ->addIndexColumn() // ✅ السطر المهم جداً
            ->addColumn('branch', fn($u) => $u->branch?->name ?? '—')
            ->addColumn('roles', function ($u) {
                return $u->roles->map(
                    fn($r) =>
                    '<span class="badge bg-info">' . $r->name . '</span>'
                )->implode(' ');
            })
            ->addColumn('actions', function ($u) {
                return '
                <button class="btn btn-sm btn-warning editUser" data-id="' . $u->id . '">تعديل</button>
        
            ';
            })
            ->rawColumns(['roles', 'actions'])
            ->make(true);
    }


    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required',
            'email' => 'required|email|unique:users',
            'password' => 'required|min:6',
            'branch_id' => 'nullable|exists:branches,id',
        ]);

        $user = User::create([
            'name' => $request->name,
            'email' => $request->email,
            'password' => Hash::make($request->password),
            'branch_id' => $request->branch_id,
        ]);

        if ($request->has('roles')) {
            $roleNames = Role::whereIn('id', $request->roles)->pluck('name')->toArray();
            $user->syncRoles($roleNames);
        }

        return response()->json(['status' => true]);
    }


    public function update(Request $request, User $user)
    {
        $request->validate([
            'name' => 'required',
            'email' => 'required|email|unique:users,email,' . $user->id,
        ]);

        $user->update([
            'name' => $request->name,
            'email' => $request->email,
            'branch_id' => $request->branch_id,
        ]);

        if ($request->filled('password')) {
            $user->update(['password' => Hash::make($request->password)]);
        }

        if ($request->has('roles')) {
            $roleNames = Role::whereIn('id', $request->roles)->pluck('name')->toArray();
            $user->syncRoles($roleNames);
        }

        return response()->json(['status' => true]);
    }

    public function edit(User $user)
    {
        return response()->json([
            'user' => $user,
            'roles' => $user->roles->pluck('id')->toArray(),
        ]);
    }



    public function destroy(User $user)
    {
        $user->delete();
        return response()->json(['status' => true]);
    }

    public function updatePassword(Request $request, $id)
    {
        try {
            $request->validate([
                'new_password' => 'required|string|min:6|confirmed',
            ]);

            $user = User::findOrFail($id);
            $user->password = Hash::make($request->new_password);
            $user->save();

            return response()->json(['success' => true, 'message' => __('messages.passwordUpdated')]);
        } catch (\Exception $e) {
            return response()->json(['error' => true, 'message' => __('flasher::messages.error')]);
        }
    }

    public function show(User $user)
    {
        $user->load('roles:id,name', 'branch:id,name');
        return response()->json([
            'id' => $user->id,
            'name' => $user->name,
            'email' => $user->email,
            'branch' => $user->branch?->name,
            'roles' => $user->roles->pluck('name'),
            'created_at' => $user->created_at?->toDateTimeString(),
            'updated_at' => $user->updated_at?->toDateTimeString(),
        ]);
    }



    // profile ***********************

    public function profile()
    {
        $profile = auth('web')->user();
        return view('admin.profile', compact('profile'));
    }

    public function update_profile(Request $request)
    {
        $validated = $request->validate([
            'name_ar' => 'required|string|max:255',
            'name_en' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email,' . auth('web')->id(),
            'password' => 'nullable|string|min:8|confirmed',
        ]);

        $user = auth('web')->user();

        // Update name in both Arabic and English
        $user->setTranslation('name', 'ar', $request->name_ar);
        $user->setTranslation('name', 'en', $request->name_en);

        // Update email
        $user->email = $request->email;

        // If a password is provided, hash it and update
        if ($request->filled('password')) {
            $user->password = bcrypt($request->password);
        }

        $user->save();

        toastr()->success(__('messages.success')); // Assuming a success message is displayed
        return redirect()->route('profile'); // Redirect to profile page
    }
}