<?php

namespace App\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\Validator;

class SaveBranchGoalsRequest extends FormRequest
{
    public function authorize(): bool
    {
        return true;
    }

    protected function prepareForValidation(): void
    {
        // ضمان وجود مصفوفة عناصر
        $this->merge([
            'items' => $this->input('items', []),
        ]);
    }

    public function rules(): array
    {
        return [
            'year'              => ['required', 'integer', 'min:2000', 'max:2100'],
            'items'             => ['required', 'array', 'size:12'], // لازم 12 شهر
            'items.*.month'     => ['required', 'integer', 'between:1,12'],
            'items.*.actual'    => ['nullable', 'numeric', 'min:0'],
            'items.*.target'    => ['nullable', 'numeric', 'min:0'],
        ];
    }

    /**
     * تحقق إضافي: تأكد أن الأشهر فريدة وتغطي 1..12
     */
    public function withValidator(Validator $validator): void
    {
        $validator->after(function (Validator $v) {
            $items = $this->input('items', []);
            $months = array_map(fn($r) => (int)($r['month'] ?? 0), $items);
            sort($months);

            // لازم تكون [1..12] بالضبط
            $expected = range(1, 12);
            if ($months !== $expected) {
                $v->errors()->add('items', 'يجب إدخال صف لكل شهر من 1 إلى 12 بدون تكرار.');
            }
        });
    }

    public function attributes(): array
    {
        return [
            'year'             => 'السنة',
            'items'            => 'الأهداف الشهرية',
            'items.*.month'    => 'الشهر',
            'items.*.actual'   => 'المبيعات الفعلية',
            'items.*.target'   => 'المبيعات المستهدفة',
        ];
    }
}