<?php

namespace App\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;

class StoreBranchRequest extends FormRequest
{
    public function authorize(): bool
    {
        return true;
    }

    /**
     * نضمن وجود مصفوفات افتراضيًا وتحو́يل بعض القيم قبل التحقق
     */
    protected function prepareForValidation(): void
    {
        $this->merge([
            'phones'  => $this->input('phones', []),
            'address' => $this->input('address', []),
            // تحويل "active" إلى boolean صريح
            'active'  => filter_var($this->input('active', true), FILTER_VALIDATE_BOOLEAN, FILTER_NULL_ON_FAILURE) ?? false,
        ]);
    }

    public function rules(): array
    {
        return [
            // الاسم المترجم
            'name.ar' => ['required', 'string', 'max:255'],
            'name.en' => ['required', 'string', 'max:255'],

            // العلاقات
            'branch_category_id' => ['required', 'exists:branch_categories,id'],
            'branch_type_id'     => ['required', 'exists:branch_types,id'],
            'city_id'            => ['required', 'exists:states,id'],

            // الهواتف
            'primary_phone'        => ['required', 'string', 'max:30', 'regex:/^[0-9+\-\s()]+$/'],
            'phones'               => ['nullable', 'array', 'max:20'],
            'phones.*.label.ar'    => ['required_with:phones.*.number', 'string', 'max:100'],
            'phones.*.label.en'    => ['required_with:phones.*.number', 'string', 'max:100'],
            'phones.*.number'      => ['required_with:phones.*.label.ar', 'string', 'max:30', 'regex:/^[0-9+\-\s()]+$/'],

            // حالة الفرع وقيود إضافية
            'active'               => ['required', 'boolean'],
            'min_age'              => ['nullable', 'integer', 'min:0', 'max:120'],
            'max_discount_percent' => ['nullable', 'numeric', 'min:0', 'max:100'],

            // العنوان (اختياري، كـ JSON)
            'address'              => ['nullable', 'array'],
            // أمثلة لو أردت تفصيلًا:
            // 'address.street.ar'  => ['nullable', 'string', 'max:255'],
            // 'address.street.en'  => ['nullable', 'string', 'max:255'],
            // 'address.district.ar'=> ['nullable', 'string', 'max:255'],
            // 'address.district.en'=> ['nullable', 'string', 'max:255'],
            // 'address.building_no'=> ['nullable', 'string', 'max:50'],
            // 'address.postal_code'=> ['nullable', 'string', 'max:20'],
            // 'address.tax_no'     => ['nullable', 'string', 'max:50'],
        ];
    }

    public function messages(): array
    {
        return [
            'primary_phone.regex'   => 'صيغة رقم الهاتف غير صحيحة.',
            'phones.*.number.regex' => 'صيغة رقم الهاتف غير صحيحة.',
        ];
    }

    public function attributes(): array
    {
        return [
            'name.ar'                => 'الاسم (ar)',
            'name.en'                => 'الاسم (en)',
            'branch_category_id'     => 'فئة الفرع',
            'branch_type_id'         => 'نوع الفرع',
            'city_id'                => 'المدينة',
            'primary_phone'          => 'الهاتف الأساسي',
            'phones.*.label.ar'      => 'مسمّى الهاتف (ar)',
            'phones.*.label.en'      => 'Label (en)',
            'phones.*.number'        => 'رقم الهاتف',
            'active'                 => 'الحالة',
            'min_age'                => 'الحد الأدنى للعمر',
            'max_discount_percent'   => 'الحد الأقصى للخصم بالنسبة المئوية',
        ];
    }
}