<?php
// app/Http/Requests/StoreServiceRequest.php
namespace App\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;

class StoreServiceRequest extends FormRequest
{
    public function authorize(): bool
    {
        return true;
    }

   

    public function rules(): array
    {
        $daysEnum = ['sat', 'sun', 'mon', 'tue', 'wed', 'thu', 'fri'];
        return [
            'name.ar' => ['required', 'string', 'max:255'],
            'name.en' => ['required', 'string', 'max:255'],

            'service_category_id' => ['required', 'exists:service_categories,id'],
            'service_type_id'     => ['required', 'exists:service_types,id'],

            'price'               => ['required', 'numeric', 'min:0'],
            'discount_price'      => ['nullable', 'numeric', 'min:0', 'lte:price'],

            'duration_value'      => ['required', 'integer', 'min:1'],
            'duration_unit'       => ['required', 'in:days,months,years'],

            'start_time'          => ['nullable', 'date_format:H:i'],
            'end_time'            => ['nullable', 'date_format:H:i'],

            'notes.ar'            => ['nullable', 'string', 'max:2000'],
            'notes.en'            => ['nullable', 'string', 'max:2000'],

            'provided_branches'   => ['nullable', 'array'],
            'provided_branches.*' => ['integer', 'exists:branches,id'],
            'sold_branches'       => ['nullable', 'array'],
            'sold_branches.*'     => ['integer', 'exists:branches,id'],

            // ===== Settings =====
            'settings'                                 => ['nullable', 'array'],

            // إيقاف
            'settings.pause.enabled'                   => ['nullable', 'boolean'],
            'settings.pause.max_times'                 => ['required_if:settings.pause.enabled,1', 'nullable', 'integer', 'min:1'],
            'settings.pause.max_duration_value'        => ['required_if:settings.pause.enabled,1', 'nullable', 'integer', 'min:1'],
            'settings.pause.max_duration_unit'         => ['required_if:settings.pause.enabled,1', 'nullable', 'in:days,months,years'],

            // تنازل
            'settings.transfer.enabled'                => ['nullable', 'boolean'],
            'settings.transfer.min_duration_value'     => ['required_if:settings.transfer.enabled,1', 'nullable', 'integer', 'min:0'],
            'settings.transfer.min_duration_unit'      => ['required_if:settings.transfer.enabled,1', 'nullable', 'in:days,months,years'],
            'settings.transfer.fee'                    => ['required_if:settings.transfer.enabled,1', 'nullable', 'numeric', 'min:0'],

            // تجديد
            'settings.renew.enabled'                   => ['nullable', 'boolean'],
            'settings.renew.with_service_id'           => ['required_if:settings.renew.enabled,1', 'nullable', 'exists:services,id'],
            'settings.renew.extra_duration_value'      => ['required_if:settings.renew.enabled,1', 'nullable', 'integer', 'min:0'],
            'settings.renew.extra_duration_unit'       => ['required_if:settings.renew.enabled,1', 'nullable', 'in:days,months,years'],
            'settings.renew.max_duration_value'        => ['required_if:settings.renew.enabled,1', 'nullable', 'integer', 'min:0'],
            'settings.renew.max_duration_unit'         => ['required_if:settings.renew.enabled,1', 'nullable', 'in:days,months,years'],

            // استرجاع
            'settings.refund.enabled'                  => ['nullable', 'boolean'],
            'settings.refund.method'                   => ['required_if:settings.refund.enabled,1', 'nullable', 'in:money,points,exchange'],
            'settings.refund.exchange_service_id'      => ['required_if:settings.refund.method,exchange', 'nullable', 'exists:services,id'],

            // أيام الأسبوع
            'settings.days.enabled'                    => ['nullable', 'boolean'],
            'settings.days.values'                     => ['required_if:settings.days.enabled,1', 'nullable', 'array', 'min:1'],
            'settings.days.values.*'                   => ['in:' . implode(',', $daysEnum)],
        ];
    }

    public function attributes(): array
    {
        return [
            'name.ar' => 'الاسم (ar)',
            'name.en' => 'الاسم (en)',
            'service_category_id' => 'فئة الخدمة',
            'service_type_id' => 'نوع الخدمة',
            'price' => 'السعر',
            'discount_price' => 'سعر الخصم',
            'duration_value' => 'المدة',
            'duration_unit' => 'نوع المدة',
        ];
    }

    protected function prepareForValidation(): void
    {
        // موجود عندك: دمج settings/notes لو حابب
        $this->merge([
            'settings' => $this->input('settings', []),
            'notes'    => $this->input('notes', []),
        ]);

        // تطبيع أوقات HH:MM حتى لو المتصفح أرسل HH:MM:SS
        foreach (['start_time', 'end_time'] as $key) {
            $val = $this->input($key);
            if ($val !== null && $val !== '') {
                // خذ أول 5 أحرف (HH:MM)
                $this->merge([$key => substr($val, 0, 5)]);
            }
        }
    }
}