<?php

namespace App\Http\Controllers;

use App\Models\State;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Yajra\DataTables\Facades\DataTables;
use App\Http\Requests\{StoreBranchRequest, UpdateBranchRequest, SaveBranchGoalsRequest};
use App\Models\{Branch, BranchCategory, BranchType, City, CompanySetting, BranchMonthlyGoal};

class BranchController extends Controller
{


    public function index(Request $request)
    {
        if ($request->ajax()) {
            $q = Branch::with(['category', 'type', 'state'])
                ->select(['id', 'name', 'branch_category_id', 'branch_type_id', 'city_id', 'primary_phone', 'active', 'created_at']);

            return DataTables::of($q)
                ->addIndexColumn()
                ->addColumn('translated_name', fn($r) => $r->getTranslation('name', app()->getLocale()))
                ->addColumn('category_name',   fn($r) => optional($r->category)?->getTranslation('name', app()->getLocale()))
                ->addColumn('type_name',       fn($r) => optional($r->type)?->getTranslation('name', app()->getLocale()))
                ->addColumn('city_name',       fn($r) => optional($r->state)?->getTranslation('name', app()->getLocale()))
                ->addColumn('active_badge',    fn($r) => $r->active
                    ? '<span class="badge bg-success">' . __('نشط') . '</span>'
                    : '<span class="badge bg-secondary">' . __('غير نشط') . '</span>')
                ->addColumn('action', function ($r) {
                    $showUrl = route('branches.show', $r->id);
                    $editUrl = route('branches.edit', $r->id);
                    return '
                    <div class="d-flex justify-content-center align-items-center gap-1">
                      <a href="' . $showUrl . '" class="btn btn-primary btn-sm" title="' . __('عرض') . '">
                        <i class="fas fa-eye"></i>
                      </a>
                      <a href="' . $editUrl . '" class="btn btn-warning btn-sm" title="' . __('تعديل') . '">
                        <i class="fas fa-edit"></i>
                      </a>
                    </div>';
                })
                ->rawColumns(['active_badge', 'action'])
                ->make(true);
        }

        return view('admin.branches.index');
    }

    /** صفحة إنشاء */
    public function create()
    {
        $countryId  = CompanySetting::value('country_id');
        $cities     = State::when($countryId, fn($q) => $q->where('country_id', $countryId))
            ->orderBy('name->' . app()->getLocale())->get();
        $categories = BranchCategory::orderBy('name->' . app()->getLocale())->get();
        $types      = BranchType::orderBy('name->' . app()->getLocale())->get();
        $suggestedCode = $this->predictNextBranchCode();

        return view('admin.branches.create', compact('cities', 'categories', 'types', 'suggestedCode'))
            ->with('branch', null);
    }

    private function predictNextBranchCode(): string
    {
        $last = DB::table('branches')
            ->select(DB::raw("MAX(CAST(branch_code AS UNSIGNED)) as seq"))
            ->value('seq');

        $next = (int) $last + 1;
        return str_pad((string) $next, 3, '0', STR_PAD_LEFT); // 001, 002, ...
    }



    /** صفحة عرض */
    public function show(Branch $branch)
    {
        $branch->load(['category', 'type', 'state']);

        return view('admin.branches.show', compact('branch'));
    }

    /** صفحة تعديل */
    public function edit(Branch $branch)
    {
        $countryId  = CompanySetting::value('country_id');
        $cities     = State::when($countryId, fn($q) => $q->where('country_id', $countryId))
            ->orderBy('name->' . app()->getLocale())->get();
        $categories = BranchCategory::orderBy('name->' . app()->getLocale())->get();
        $types      = BranchType::orderBy('name->' . app()->getLocale())->get();

        return view('admin.branches.create', compact('branch', 'cities', 'categories', 'types'));
    }

    /** تحديث + أهدافه ثم رجوع للـ Index */
    public function update(UpdateBranchRequest $request, Branch $branch)
    {
        DB::transaction(function () use ($request, $branch) {
            $phones = collect($request->input('phones', []))
                ->filter(fn($row) => isset($row['number']) && trim((string)$row['number']) !== '')
                ->map(fn($row) => [
                    'label' => ['ar' => $row['label']['ar'] ?? '', 'en' => $row['label']['en'] ?? ''],
                    'number' => $row['number'],
                ])->values()->all();

            $branch->update([
                'name'                => $request->input('name'),
                'branch_category_id'  => $request->branch_category_id,
                'branch_type_id'      => $request->branch_type_id,
                'city_id'             => $request->city_id,
                'primary_phone'       => $request->primary_phone,
                'phones'              => $phones,
                'active'              => (bool)$request->active,
                'address'             => $request->input('address', []),
                'min_age'             => $request->input('min_age'),
                'max_discount_percent' => $request->input('max_discount_percent'),
                'short_name'          => $request->input('short_name'),
                'gender'              => (int) $request->input('gender'),
            ]);

            $this->upsertGoalsFromRequest($request, $branch->id);
        });

        return redirect()->route('branches.index')->with('success', __('تم تحديث الفرع بنجاح.'));
    }

    /** جلب أهداف سنة معينة (للعرض/التعديل) */
    public function goals(Branch $branch, Request $request)
    {
        $year = (int)($request->query('year') ?: now()->year);
        $rows = $branch->goals()->where('year', $year)->get()->keyBy('month');

        $items = [];
        for ($m = 1; $m <= 12; $m++) {
            $items[] = [
                'month'  => $m,
                'actual' => optional($rows->get($m))->actual,
                'target' => optional($rows->get($m))->target,
            ];
        }
        return response()->json(['year' => $year, 'items' => $items]);
    }

    public function store(StoreBranchRequest $request)
    {
        DB::transaction(function () use ($request, &$branch) {
            // (1) كود الفرع التسلسلي
            $nextCode = $this->generateNextBranchCode();

            // (2) ترتيب/تنقية الهواتف
            $phones = collect($request->input('phones', []))
                ->filter(fn($row) => isset($row['number']) && trim((string)$row['number']) !== '')
                ->map(fn($row) => [
                    'label'  => ['ar' => $row['label']['ar'] ?? '', 'en' => $row['label']['en'] ?? ''],
                    'number' => $row['number'],
                ])->values()->all();

            // (3) إنشاء الفرع
            $branch = Branch::create([
                'branch_code'          => $nextCode,
                'name'                 => $request->input('name'),
                'branch_category_id'   => $request->branch_category_id,
                'branch_type_id'       => $request->branch_type_id,
                'city_id'              => $request->city_id, // مربوط بـ states(id)
                'primary_phone'        => $request->primary_phone,
                'phones'               => $phones,
                'active'               => (bool) $request->active,
                'address'              => $request->input('address', []),
                'min_age'              => $request->input('min_age'),
                'max_discount_percent' => $request->input('max_discount_percent'),

                'short_name'           => $request->input('short_name'),
                'gender'               => (int) $request->input('gender'), // 0=رجال, 1=نساء
            ]);

            // (4) حفظ الأهداف (المستهدف فقط)، الفعلية = 0
            $this->upsertGoalsFromRequest($request, $branch->id);
        });

        return redirect()->route('branches.index')->with('success', __('dashboard.created_success'));
    }

    /** يولّد الكود التالي بشكل آمن داخل الترانزاكشن */
    private function generateNextBranchCode(): string
    {
        $last = DB::table('branches')
            ->lockForUpdate()
            ->select(DB::raw("MAX(CAST(branch_code AS UNSIGNED)) as seq"))
            ->value('seq');

        $next = (int) $last + 1;
        return str_pad((string)$next, 3, '0', STR_PAD_LEFT); // 001, 002, ...
    }

    /** حفظ/تحديث الأهداف من نفس الطلب (target فقط) و actual=0 */
    private function upsertGoalsFromRequest(Request $request, int $branchId): void
    {
        $year  = (int) ($request->input('goals.year') ?? now()->year);
        $items = collect($request->input('goals.items', []));

        $payload = $items->map(function ($row) use ($branchId, $year) {
            $month  = (int) ($row['month'] ?? 0);
            $target = $row['target'] ?? null;

            if ($month < 1 || $month > 12) return null;
            if ($target === null || $target === '') return null;

            return [
                'branch_id'  => $branchId,
                'year'       => $year,
                'month'      => $month,
                'target'     => (float) $target,
                'actual'     => 0.0,        // مخفية
                'created_at' => now(),
                'updated_at' => now(),
            ];
        })
            ->filter()
            ->values()
            ->all();

        if (empty($payload)) {
            return;
        }

        // unique: (branch_id,year,month)
        BranchMonthlyGoal::upsert(
            $payload,
            ['branch_id', 'year', 'month'],
            ['target', 'actual', 'updated_at']
        );
    }
}
