<?php

namespace App\Http\Controllers;

use App\Models\Exam;
use App\Models\Semester;
use Illuminate\Http\Request;
use Yajra\DataTables\DataTables;

class ExamController extends Controller
{
    public function store(Request $request)
    {
      
        $request->validate(
            [
                'lesson_id' => 'required|exists:lessons,id',
                'title_ar' => 'required|string|max:255',
                'title_en' => 'required|string|max:255',
                'description_ar' => 'nullable|string',
                'description_en' => 'nullable|string',
                'min_time' => 'required|integer|min:1',
                'max_time' => 'required|integer|min:1|gte:min_time', // Ensure max_time is greater than or equal to min_time
                'average_time' => 'required|integer|min:1|gte:min_time|lte:max_time', // Ensure average_time is between min_time and max_time
            ],
            [
                'gte' => __('dashboard.gte'),
                'lte' => __('dashboard.lte'),
            ],
        );

        // Create the exam and set the translations for title and description
        $exam = Exam::create([
            'lesson_id' => $request->lesson_id,
            'min_time' => $request->min_time,
            'max_time' => $request->max_time,
            'average_time' => $request->average_time,

            'title' => [
                'ar' => $request->title_ar,
                'en' => $request->title_en,
            ],
            'description' => [
                'ar' => $request->description_ar,
                'en' => $request->description_en,
            ],
        ]);

        toastr()->success(__('messages.success'));
        return redirect()->back();
    }

    // Update an existing exam
    public function update(Request $request, $id)
    {
        // Validate the incoming request
        $request->validate([
            'title_ar' => 'required|string|max:255',
            'title_en' => 'required|string|max:255',
            'description_ar' => 'nullable|string',
            'description_en' => 'nullable|string',
            'min_time' => 'nullable|integer|min:0',
            'max_time' => 'nullable|integer|min:0',
            'average_time' => 'nullable|integer|min:0',
        ]);

        // Find the exam by its ID
        $exam = Exam::findOrFail($id);

        // Update the exam with the new data
        $exam->update([
            'title' => [
                'ar' => $request->title_ar,
                'en' => $request->title_en,
            ],
            'description' => [
                'ar' => $request->description_ar,
                'en' => $request->description_en,
            ],
            'min_time' => $request->min_time,
            'max_time' => $request->max_time,
            'average_time' => $request->average_time,
        ]);

        toastr()->success(__('messages.success'));
        return redirect()->back();
    }

    public function getExamsData($subjectId)
    {
        // Query the exams for the subject
        $examsQuery = Exam::whereHas('lesson.unit.semester.subject', function ($query) use ($subjectId) {
            $query->where('id', $subjectId);
        })
            ->with(['lesson.unit.semester']) // Eager load relationships
            ->select('exams.*'); // Select the required columns

        // Handle the filtering logic
        if (request()->has('semester') && request()->semester != '') {
            $examsQuery->whereHas('lesson.unit.semester', function ($query) {
                $query->where('id', request()->semester);
            });
        }

        if (request()->has('unit') && request()->unit != '') {
            $examsQuery->whereHas('lesson.unit', function ($query) {
                $query->where('id', request()->unit);
            });
        }
        // Global search (searching across all columns)
        if (request()->has('search') && request()->search['value'] != '') {
            $searchTerm = request()->search['value'];
            $examsQuery->where(function ($query) use ($searchTerm) {
                $query
                    ->where('title->ar', 'like', '%' . $searchTerm . '%')
                    ->orWhere('title->en', 'like', '%' . $searchTerm . '%')
                    ->orWhereHas('lesson', function ($q) use ($searchTerm) {
                        $q->where('title->ar', 'like', '%' . $searchTerm . '%')->orWhere('title->en', 'like', '%' . $searchTerm . '%');
                    })
                    ->orWhereHas('lesson.unit', function ($q) use ($searchTerm) {
                        $q->where('name->ar', 'like', '%' . $searchTerm . '%')->orWhere('name->en', 'like', '%' . $searchTerm . '%');
                    })
                    ->orWhereHas('lesson.unit.semester', function ($q) use ($searchTerm) {
                        $q->where('name->ar', 'like', '%' . $searchTerm . '%')->orWhere('name->en', 'like', '%' . $searchTerm . '%');
                    })
                    ->orWhere('min_time', 'like', '%' . $searchTerm . '%')
                    ->orWhere('max_time', 'like', '%' . $searchTerm . '%')
                    ->orWhere('average_time', 'like', '%' . $searchTerm . '%');
            });
        }

        if (request()->has('order') && is_array(request()->order)) {
            $orderColumn = request()->order[0]['column']; // Column index for sorting
            $orderDir = request()->order[0]['dir']; // Sorting direction (asc or desc)

            // The columns to order by
            $columns = ['title', 'lesson', 'unit', 'semester'];

            // Switch between columns
            switch ($orderColumn) {
                case 0: // Sorting by exam title
                    $examsQuery->orderBy('exams.title', $orderDir);
                    break;

                case 1: // Sorting by lesson title
                    $examsQuery->join('lessons', 'lessons.id', '=', 'exams.lesson_id')->orderBy('lessons.title', $orderDir);
                    break;

                case 2: // Sorting by unit name
                    $examsQuery->join('lessons', 'lessons.id', '=', 'exams.lesson_id')->join('units', 'units.id', '=', 'lessons.unit_id')->orderBy('units.name', $orderDir);
                    break;

                case 3: // Sorting by semester name
                    $examsQuery->join('lessons', 'lessons.id', '=', 'exams.lesson_id')->join('units', 'units.id', '=', 'lessons.unit_id')->join('semesters', 'semesters.id', '=', 'units.semester_id')->orderBy('semesters.name', $orderDir);
                    break;

                default:
                    $examsQuery->orderBy('exams.title', $orderDir); // Default ordering by title
                    break;
            }
        }

        // Return the data to the DataTable
        return DataTables::of($examsQuery)
            ->addColumn('title', function ($exam) {
                return $exam->getTranslation('title', app()->getLocale());
            })
            ->addColumn('min_time', function ($exam) {
                return $exam->min_time;
            })
            ->addColumn('max_time', function ($exam) {
                return $exam->max_time;
            })
            ->addColumn('average_time', function ($exam) {
                return $exam->average_time;
            })
            ->addColumn('action', function ($exam) {
                // return '<button type="button" class="btn btn-sm btn-danger delete-exam" data-id="' . $exam->id . '">Delete</button>';
            })
            ->addColumn('lesson', function ($exam) {
                return $exam->lesson->getTranslation('title', app()->getLocale());
            })
            ->addColumn('unit', function ($exam) {
                return $exam->lesson->unit->getTranslation('name', app()->getLocale());
            })
            ->addColumn('semester', function ($exam) {
                return $exam->lesson->unit->semester->getTranslation('name', app()->getLocale());
            })
            ->make(true);
    }

    // Delete exam
    public function destroy($id)
    {
        $exam = Exam::findOrFail($id);
        $exam->delete();

        return response()->json(['success' => 'Exam deleted successfully']);
    }
}
