<?php

namespace App\Http\Controllers;

use App\Models\Grade;
use App\Models\Classroom;
use Illuminate\Http\Request;
use Yajra\DataTables\DataTables;
use Illuminate\Support\Facades\DB;

class GradeController extends Controller
{
  public function index(Request $request)
{
    if ($request->ajax()) {
        $grades = Grade::with('classrooms')->select(['id', 'name']);

        return DataTables::of($grades)
            ->filter(function ($query) use ($request) {
                if ($search = $request->get('search')['value']) {
                    $query->where(function ($q) use ($search) {
                        $q->where('name->ar', 'LIKE', "%{$search}%")
                          ->orWhere('name->en', 'LIKE', "%{$search}%")
                          ->orWhereHas('classrooms', function ($classQuery) use ($search) {
                              $classQuery->where('name->ar', 'LIKE', "%{$search}%")
                                         ->orWhere('name->en', 'LIKE', "%{$search}%");
                          });
                    });
                }
            })
            ->addColumn('translated_name', function ($row) {
                return $row->getTranslation('name', app()->getLocale());
            })
            ->addColumn('classrooms', function ($row) {
                if ($row->classrooms->isEmpty()) {
                    return '<span class="badge bg-danger text-wrap">' . __('dashboard.no_classes_found') . '</span>';
                }

                $classDropdown = '<div class="btn-group">
                                    <button class="btn btn-info btn-sm dropdown-toggle" type="button" data-bs-toggle="dropdown">
                                        ' . __('dashboard.showClass') . '
                                    </button>
                                    <ul class="dropdown-menu text-center">';

                foreach ($row->classrooms as $class) {
                    $classDropdown .= '<li>
                        <a class="dropdown-item text-primary edit-class-btn" href="#" 
                            data-id="' . $class->id . '" 
                            data-name_ar="' . $class->getTranslation('name', 'ar') . '" 
                            data-name_en="' . $class->getTranslation('name', 'en') . '" 
                            data-bs-toggle="modal" 
                            data-bs-target="#editClassModal">
                            <i class="fas fa-edit mx-2"></i> ' . $class->getTranslation('name', app()->getLocale()) . '
                        </a>
                    </li>';
                }

                $classDropdown .= '</ul></div>';

                return $classDropdown;
            
            })

            
            ->addColumn('action', function ($row) {
                return '
                    <div class="btn-group">
                        <button class="btn btn-success dropdown-toggle btn-sm btn-pill" type="button" data-bs-toggle="dropdown" aria-expanded="false">
                            <i class="fas fa-cogs"></i> ' . __('dashboard.action') . '
                        </button>
                        <ul class="dropdown-menu text-center">
                            <li>
                                <a class="dropdown-item text-warning edit-btn" href="#" 
                                    data-id="' . $row->id . '" 
                                    data-name_ar="' . $row->getTranslation('name', 'ar') . '" 
                                    data-name_en="' . $row->getTranslation('name', 'en') . '" 
                                    data-bs-toggle="modal" 
                                    data-bs-target="#editGradeModal">
                                    <i class="fas fa-edit mx-2"></i> ' . __('dashboard.editGrade') . '
                                </a>
                            </li>
                            <li>
                                <a class="dropdown-item text-primary add-repeater-btn" href="#" 
                                    data-bs-toggle="modal" 
                                      data-grade_id="' . $row->id . '" 
                                    data-bs-target="#addClassToGrade">
                                    <i class="fas fa-plus-circle mx-2"></i> ' . __('dashboard.addClass') . '
                                </a>
                            </li>
                        </ul>
                    </div>
                ';
            })
            ->rawColumns(['action', 'classrooms'])
            ->make(true);
    }

    return view('admin.pages.grade.index');
}


    public function store(Request $request)
    {
        $request->validate([
            'grades' => 'required|array',
            'grades.*.name_ar' => 'required|string|max:255|unique:grades,name->ar',
            'grades.*.name_en' => 'required|string|max:255|unique:grades,name->en',
        ]);

        foreach ($request->grades as $grade) {
            Grade::create([
                'name' => [
                    'ar' => $grade['name_ar'],
                    'en' => $grade['name_en'],
                ],
            ]);
        }

        // Show success message and redirect
        toastr()->success(__('messages.success'));
        return redirect()->route('grades.index');
    }

    public function update(Request $request, $id)
    {
        $request->validate([
            'name' => 'required|array',
            'name.ar' => 'required|string',
            'name.en' => 'required|string',
        ]);

        $grade = Grade::findOrFail($id);
        $grade->setTranslation('name', 'ar', $request->name['ar']);
        $grade->setTranslation('name', 'en', $request->name['en']);
        $grade->save();

        toastr()->success(__('messages.success'));

        return response()->json(['success' => true, 'message' => __('messages.success')]);
    }

    public function class_store(Request $request)
    {

      
        $request->validate([
            'class' => 'required|array',
            'class.*' => 'required|array',
            'class.*.name_ar' => 'required|string|max:255|unique:classrooms,name->ar',
            'class.*.name_en' => 'required|string|max:255|unique:classrooms,name->en',
            'grade_id' => 'required|exists:grades,id',
        ]);

        try {
            DB::transaction(function () use ($request) {
                foreach ($request->class as $item) {
                    Classroom::create([
                        'name' => [
                            'ar' => $item['name_ar'],
                            'en' => $item['name_en'],
                        ],
                        'grade_id' => $request->grade_id,
                    ]);
                }
            });

            toastr()->success(__('messages.success'));
            return back();
        } catch (\Exception $e) {
            toastr()->error(__('messages.error'));
            return back()->withErrors(['error' => $e->getMessage()]);
        }
    }


    public function update_class(Request $request, $id)
{


    $request->validate([
        'name_ar' => 'required|string|max:255',
        'name_en' => 'required|string|max:255',
    ]);

    $class = Classroom::findOrFail($id);
    $class->setTranslation('name', 'ar', $request->name_ar);
    $class->setTranslation('name', 'en', $request->name_en);
    $class->save();

    return response()->json(['message' => __('messages.success')]);
}



}
