<?php

namespace App\Http\Controllers;

use Carbon\Carbon;
use Stripe\Stripe;
use App\Models\Offer;
use App\Models\Payment;
use App\Models\Subject;
use App\Models\Subscription;
use Illuminate\Http\Request;
use Stripe\Checkout\Session;
use App\Models\MemberPayment;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Http;

class PaymentController extends Controller
{

    public function start(Offer $offer)
    {
        $member = auth('member')->user();

        $amount = $offer->items->sum('final_price');

        if ($amount <= 0) {
            return back()->with('error', __('member.invalid_amount'));
        }

        $payment = MemberPayment::create([
            'member_id' => $member->id,
            'branch_id' => $member->branch_id,
            'item_type' => 'offer',
            'item_id'   => $offer->id,
            'item_name' => $offer->name['ar'] ?? 'Offer',
            'amount'    => $amount,
            'currency'  => 'SAR',
            'merchant_transaction_id' => Str::uuid(),
            'status' => 'pending',
        ]);

        return redirect()->route('member.payment.checkout', $payment);
    }


    public function checkout(MemberPayment $payment)
    {
        $member = auth('member')->user();

        abort_if($payment->member_id !== $member->id, 403);

        if (!$payment->checkout_id) {

            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . config('hyperpay.token'),
            ])->asForm()->post(
                config('hyperpay.base_url') . '/v1/checkouts',
                [
                    'entityId' => config('hyperpay.entity_id'),
                    'amount'   => number_format($payment->amount, 2, '.', ''),
                    'currency' => 'SAR',
                    'paymentType' => 'DB',

                    'merchantTransactionId' => $payment->merchant_transaction_id,
                    'testMode' => 'EXTERNAL',

                    'customer.email' => $member->email,
                    'customer.givenName' => $member->name,
                    'customer.surname' => $member->name,

                    'billing.street1' => 'Test street',
                    'billing.city' => 'Riyadh',
                    'billing.state' => 'Riyadh',
                    'billing.country' => 'SA',
                    'billing.postcode' => '12345',
                ]
            );

            if (!$response->successful()) {
                abort(500, 'HyperPay connection failed');
            }

            $payment->update([
                'checkout_id' => $response->json('id'),
            ]);
        }

        return view('members.payments.checkout', compact('payment'));
    }


    public function result(Request $request)
    {
        $resourcePath = $request->query('resourcePath');

        if (!$resourcePath) {
            return redirect()->route('member.dashboard')
                ->with('error', __('member.payment_failed'));
        }

        $response = Http::withHeaders([
            'Authorization' => 'Bearer ' . config('hyperpay.token'),
        ])->get(config('hyperpay.base_url') . $resourcePath, [
            'entityId' => config('hyperpay.entity_id'),
        ]);

        $data = $response->json();

        // dd($data['result']);


        $payment = MemberPayment::where(
            'merchant_transaction_id',
            data_get($data, 'merchantTransactionId')
        )->first();

        if (!$payment) {
            return redirect()->route('member.dashboard')
                ->with('error', __('member.payment_failed'));
        }

        $code = data_get($data, 'result.code');

        // ✅ SUCCESS (الشرط الصح)
        if ($code && str_starts_with($code, '000.')) {

            DB::transaction(function () use ($payment, $data) {

                $payment->update([
                    'status' => 'paid',
                    'transaction_id' => data_get($data, 'id'),
                    'result_code' => data_get($data, 'result.code'),
                    'result_description' => data_get($data, 'result.description'),
                ]);

                app(\App\Services\SubscriptionService::class)
                    ->createFromPayment($payment);
            });

            return redirect()
                ->route('member.my_subscriptions')
                ->with('success', __('member.payment_success'));
        }

        // ❌ FAILED
        $payment->update([
            'status' => 'failed',
            'result_code' => $code,
            'result_description' => data_get($data, 'result.description'),
        ]);

        return redirect()->route('member.dashboard')
            ->with('error', __('member.payment_failed'));
    }


    public function my_subscriptions(Request $request)
    {
        $member = auth('member')->user();

        $query = \App\Models\Subscription::query()
            ->where('member_id', $member->id);

        // 🔍 البحث بالاسم
        if ($request->filled('search')) {
            $query->where('item_name', 'LIKE', '%' . trim($request->search) . '%');
        }

        // 🎯 فلتر الحالة
        if ($request->filled('status')) {
            if ($request->status === 'active') {
                $query->whereDate('end_date', '>=', now());
            }

            if ($request->status === 'expired') {
                $query->whereDate('end_date', '<', now());
            }
        }

        // ⬇️ الترتيب
        if ($request->sort === 'ending_soon') {
            $query->orderBy('end_date', 'asc');
        } else {
            $query->orderBy('id', 'desc');
        }

        $subscriptions = $query->paginate(9);

        return view('members.subscriptions.index', compact('subscriptions'));
    }
}
