<?php


namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\PaymentMethod;
use Yajra\DataTables\Facades\DataTables;
use Illuminate\Database\QueryException;

class PaymentMethodController extends Controller
{
    public function index(Request $request)
    {
        if ($request->ajax()) {
            $q = PaymentMethod::select(['id', 'name_ar', 'name_en', 'code', 'active', 'created_at']);

            return DataTables::of($q)
                ->addIndexColumn()
                ->editColumn(
                    'active',
                    fn($r) => $r->active
                        ? '<span class="badge bg-success">نشط</span>'
                        : '<span class="badge bg-secondary">غير نشط</span>'
                )
                ->editColumn('created_at', fn($r) => $r->created_at->format('Y-m-d H:i'))
                ->addColumn('action', fn($r) => '
                    <button class="btn btn-warning btn-sm editMethod" data-id="' . $r->id . '"><i class="fas fa-edit"></i></button>
                    <button class="btn btn-danger btn-sm deleteMethod" data-id="' . $r->id . '"><i class="fas fa-trash"></i></button>
                ')
                ->rawColumns(['active', 'action'])
                ->make(true);
        }

        return view('admin.payment_methods.index');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name_ar' => 'required|string|max:255',
            'name_en' => 'required|string|max:255',
            'code' => 'nullable|string|max:50',
        ]);

        PaymentMethod::updateOrCreate(
            ['id' => $request->id],
            [
                'name_ar' => $request->name_ar,
                'name_en' => $request->name_en,
                'code' => $request->code,
                'active' => $request->has('active'),
            ]
        );

        return response()->json(['success' => true, 'message' => 'تم الحفظ بنجاح ✅']);
    }

    public function edit($id)
    {
        return response()->json(PaymentMethod::findOrFail($id));
    }

    public function destroy($id)
    {
        try {
            $method = PaymentMethod::findOrFail($id);

            // تحقق من العلاقة
            if ($method->subscriptions()->exists()) {
                return response()->json([
                    'success' => false,
                    'message' => '❌ لا يمكن حذف هذه الطريقة لأنها مستخدمة في اشتراكات.'
                ], 400);
            }

            $method->delete();
            return response()->json(['success' => true, 'message' => 'تم الحذف بنجاح ✅']);
        } catch (QueryException $e) {
            return response()->json([
                'success' => false,
                'message' => 'حدث خطأ أثناء الحذف: ' . $e->getMessage()
            ], 500);
        }
    }
}