<?php

namespace App\Http\Controllers;

use App\Models\Subject;
use Illuminate\Http\Request;
use App\Models\QuestionGroup;
use Yajra\DataTables\DataTables;

class QuestionGroupController extends Controller
{
  public function getQuestionGroupsData($subjectId)
  {
      // Query the question groups for the specific subject
      $questionGroupsQuery = QuestionGroup::where('subject_id', $subjectId)
          ->select('id', 'name', 'description', 'subject_id'); // Only select needed columns
  
      // Return data to the DataTable
      return DataTables::of($questionGroupsQuery)
          ->addColumn('name', function ($questionGroup) {
              // Retrieve the translated 'name' field depending on the locale
              return $questionGroup->getTranslation('name', app()->getLocale());
          })
          ->addColumn('description', function ($questionGroup) {
            // Retrieve the translated 'name' field depending on the locale
            return $questionGroup->getTranslation('description', app()->getLocale());
        })
        ->addColumn('action', function ($questionGroup) {
          return "<button
          data-id='{$questionGroup->id}'
          data-name_ar='{$questionGroup->getTranslation('name', 'ar')}'
          data-name_en='{$questionGroup->getTranslation('name', 'en')}'
          data-description_ar='{$questionGroup->getTranslation('description', 'ar')}'
          data-description_en='{$questionGroup->getTranslation('description', 'en')}'
          class='btn btn-info shadow editGroup-btn btn-sm' data-bs-toggle='modal' data-bs-target='#editGroup'> <i
                      class='fas fa-edit mx-2'></i>
          " .
              __('dashboard.edit') .
              "
      </button>";
      })
          ->make(true);
  }

  public function store(Request $request)
  {

      // Validate the incoming request
      $request->validate([
          'name_ar' => 'required|string|max:255',  // Arabic name is required
          'name_en' => 'required|string|max:255',  // English name is required
          'description_ar' => 'nullable|string',   // Arabic description is optional
          'description_en' => 'nullable|string',   // English description is optional
      ]);
  
      // Create a new QuestionGroup and store translations automatically
      $questionGroup = QuestionGroup::create([
          'subject_id' => $request->subject_id,
          'name' => [
              'ar' => $request->name_ar,
              'en' => $request->name_en,
          ],
          'description' => [
              'ar' => $request->description_ar,
              'en' => $request->description_en,
          ],
      ]);
  
      // Flash a success message
      toastr()->success(__('messages.success'));
  
      // Redirect back
      return redirect()->back();
  }


  public function update(Request $request, $id)
{

  
    // Validate the incoming request
    $request->validate([
        'name_ar' => 'required|string|max:255',
        'name_en' => 'required|string|max:255',
        'description_ar' => 'nullable|string',
        'description_en' => 'nullable|string',
    ]);

    // Find the QuestionGroup
    $questionGroup = QuestionGroup::findOrFail($id);

    // Update the QuestionGroup with the new translated data
    $questionGroup->update([
        'name' => [
            'ar' => $request->name_ar,
            'en' => $request->name_en,
        ],
        'description' => [
            'ar' => $request->description_ar,
            'en' => $request->description_en,
        ],
    ]);

    // Redirect back with a success message
    toastr()->success(__('messages.success'));
    return redirect()->back();
}

  
  

}
