<?php

namespace App\Http\Controllers;

use Carbon\Carbon;
use App\Models\User;
use App\Models\Branch;
use App\Models\Subscription;
use Illuminate\Http\Request;
use App\Models\ServiceInvoice;
use Yajra\DataTables\DataTables;

class ReportController extends Controller
{


    // ====== صفحة التقرير ======
    public function serviceInvoicesReport(Request $request)
    {
        return view('reports.service_invoices', [
            'branches'  => Branch::orderBy('name')->get(['id', 'name']),
            'providers' => User::orderBy('name')->get(['id', 'name']),
        ]);
    }

    // ====== DataTables JSON ======
    public function serviceInvoicesData(Request $request)
    {
        $query = ServiceInvoice::query()
            ->with([
                'member:id,name',
                'branch:id,name',

            ]);

        $this->applyFilters($query, $request);

        return DataTables::of($query)
            ->addIndexColumn()
            ->addColumn('member_name', fn($r) => $r->member->name ?? '—')
            ->addColumn('branch_name', fn($r) => $r->branch->name ?? '—')

            ->editColumn('total_net', fn($r) => number_format((float)$r->total_net, 2))
            ->make(true);
    }

    // ====== Stats JSON ======
    public function serviceInvoicesStats(Request $request)
    {
        $query = ServiceInvoice::query();
        $this->applyFilters($query, $request);

        return response()->json([
            'count'        => (int) (clone $query)->count(),
            'sum_net'      => round((float) (clone $query)->sum('total_net'), 2),
            'sum_tax'      => round((float) (clone $query)->sum('total_tax'), 2),
            'sum_discount' => round((float) (clone $query)->sum('total_discount'), 2),
        ]);
    }

    // ====== صفحة الطباعة (HTML جاهز) ======
    public function serviceInvoicesPrint(Request $request)
    {
        $query = ServiceInvoice::query()
            ->with([
                'member:id,name',
                'branch:id,name',

            ]);

        $this->applyFilters($query, $request);

        $rows = $query->orderBy('id', 'desc')->get();

        $stats = [
            'count'        => $rows->count(),
            'sum_net'      => round((float) $rows->sum('total_net'), 2),
            'sum_tax'      => round((float) $rows->sum('total_tax'), 2),
            'sum_discount' => round((float) $rows->sum('total_discount'), 2),
        ];

        return view('reports.service_invoices_print', [
            'rows'  => $rows,
            'stats' => $stats,
            'from'  => $request->date_from,
            'to'    => $request->date_to,
        ]);
    }

    // ====== فلترة مشتركة (مصدر الحقيقة) ======
    private function applyFilters($query, Request $request): void
    {
        $from = $request->filled('date_from')
            ? Carbon::parse($request->date_from)->startOfDay()
            : null;

        $to = $request->filled('date_to')
            ? Carbon::parse($request->date_to)->endOfDay()
            : null;

        if ($from) $query->where('created_at', '>=', $from);
        if ($to)   $query->where('created_at', '<=', $to);

        if ($request->filled('branch_id')) {
            $query->where('branch_id', $request->branch_id);
        }
    }


    /* ========= صفحة التقرير ========= */
    public function index()
    {
        return view('reports.subscriptions_report', [
            'branches' => Branch::orderBy('name')->get(['id', 'name']),
            'users'    => User::orderBy('name')->get(['id', 'name']),
        ]);
    }

    /* ========= DataTables ========= */
    public function data(Request $request)
    {
        $q = Subscription::query()
            ->with([
                'member:id,name',
                'branch:id,name',
              
            ]);

        $this->applyFiltersSub($q, $request);

        return DataTables::of($q)
            ->addIndexColumn()
            ->addColumn('member_name', fn($r) => $r->member->name ?? '—')
            ->addColumn('branch_name', fn($r) => $r->branch->name ?? '—')
       
            ->editColumn('total', fn($r) => number_format((float)$r->total, 2))
            ->make(true);
    }

    /* ========= الإحصائيات ========= */
    public function stats(Request $request)
    {
        $q = Subscription::query();
        $this->applyFiltersSub($q, $request);

        return response()->json([
            'count'   => (clone $q)->count(),
            'sum'     => round((float)(clone $q)->sum('total'), 2),
            'tax'     => round((float)(clone $q)->sum('tax'), 2),
            'discount' => round((float)(clone $q)->sum('discount'), 2),
        ]);
    }

    /* ========= الطباعة ========= */
    public function print(Request $request)
    {
        $q = Subscription::query()
            ->with(['member', 'branch']);

        $this->applyFiltersSub($q, $request);

        $rows = $q->orderBy('id', 'desc')->get();

        return view('reports.subscriptions_print', [
            'rows' => $rows,
            'from' => $request->date_from,
            'to'   => $request->date_to,
            'stats' => [
                'count'    => $rows->count(),
                'sum'      => $rows->sum('total'),
                'tax'      => $rows->sum('tax'),
                'discount' => $rows->sum('discount'),
            ]
        ]);
    }

    /* ========= فلترة مشتركة ========= */
    private function applyFiltersSub($q, Request $r)
    {
        if ($r->filled('date_from')) {
            $q->whereDate('created_at', '>=', Carbon::parse($r->date_from));
        }

        if ($r->filled('date_to')) {
            $q->whereDate('created_at', '<=', Carbon::parse($r->date_to));
        }

        if ($r->filled('branch_id')) {
            $q->where('branch_id', $r->branch_id);
        }

        if ($r->filled('created_by')) {
            $q->where('created_by', $r->created_by);
        }

        if ($r->filled('status')) {
            $q->where('status', $r->status);
        }

        if ($r->filled('type')) {
            $q->where('subscription_type', $r->type);
        }
    }
}