<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Role;
use App\Models\Permission;
use Yajra\DataTables\Facades\DataTables;

class RoleController extends Controller
{
    public function index()
    {
        $permissions = Permission::select('id', 'name')->get();
        return view('roles.index', compact('permissions'));
    }

    public function list(Request $request)
    {
        $query = Role::with('permissions');

        if ($request->name) {
            $query->where('name', 'like', "%{$request->name}%");
        }

        $roles = $query->get();

        return DataTables::of($roles)
            ->addIndexColumn()
            ->addColumn('permissions', function ($role) {
                return '<button class="btn btn-primary btn-sm viewPerms" 
                data-role="' . $role->id . '"
                data-name="' . $role->name . '">
                عرض الصلاحيات
            </button>';
            })

            ->addColumn('actions', function ($role) {
                return '
                    <button class="btn btn-sm btn-warning editRole" data-id="' . $role->id . '">تعديل</button>
                    <button class="btn btn-sm btn-danger deleteRole" data-id="' . $role->id . '">حذف</button>
                ';
            })
            ->rawColumns(['permissions', 'actions'])
            ->make(true);
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|unique:roles,name',
        ]);

        $role = Role::create([
            'name' => $request->name,
            'guard_name' => 'web',
        ]);

        // ✅ حول IDs إلى أسماء قبل syncPermissions
        if ($request->has('permissions')) {
            $permissionNames = Permission::whereIn('id', $request->permissions)->pluck('name')->toArray();
            $role->syncPermissions($permissionNames);
        }

        return response()->json(['success' => true]);
    }

    public function update(Request $request, Role $role)
    {
        $request->validate([
            'name' => 'required',
        ]);

        $role->update(['name' => $request->name]);

        // ✅ نفس الفكرة هنا
        if ($request->has('permissions')) {
            $permissionNames = Permission::whereIn('id', $request->permissions)->pluck('name')->toArray();
            $role->syncPermissions($permissionNames);
        } else {
            $role->syncPermissions([]); // لو مفيش صلاحيات مرسلة، فضيها
        }

        return response()->json(['success' => true]);
    }

    public function edit(Role $role)
    {
        return response()->json([
            'role' => $role,
            'permissions' => $role->permissions->pluck('id')->toArray(),
            'permissions_details' => $role->permissions->map(function ($p) {
                return ['id' => $p->id, 'name' => $p->name];
            })
        ]);
    }




    public function destroy(Role $role)
    {
        $role->delete();
        return response()->json(['success' => true]);
    }
    public function show(Role $role)
    {
        $role->load('permissions:id,name');
        return response()->json([
            'id' => $role->id,
            'name' => $role->name,
            'permissions' => $role->permissions->pluck('name'),
            'created_at' => $role->created_at?->toDateTimeString(),
            'updated_at' => $role->updated_at?->toDateTimeString(),
        ]);
    }
}