<?php

namespace App\Http\Controllers;

use App\Http\Requests\StoreServiceCategoryRequest;
use App\Http\Requests\UpdateServiceCategoryRequest;
use App\Models\ServiceCategory;
use Illuminate\Http\Request;
use Yajra\DataTables\Facades\DataTables;

class ServiceCategoryController extends Controller
{
    /** Index: View + DataTables JSON */
    public function index(Request $request)
    {
        if ($request->ajax()) {
            $q = ServiceCategory::query()->select(['id','name','active','created_at']);

            return DataTables::of($q)
                ->addIndexColumn()
                ->addColumn('translated_name', fn(ServiceCategory $r) => $r->getTranslation('name', app()->getLocale()))
                ->addColumn('active_badge', fn(ServiceCategory $r) =>
                    $r->active ? '<span class="badge bg-success">نشط</span>'
                               : '<span class="badge bg-secondary">غير نشط</span>'
                )
                ->addColumn('action', function (ServiceCategory $r) {
                    $showUrl = route('service-categories.show', $r->id);
                    $editUrl = route('service-categories.edit', $r->id);
                    return '
                      <div class="d-flex justify-content-center align-items-center gap-1">
                        <a href="'.$showUrl.'" class="btn btn-primary btn-sm" title="عرض"><i class="fas fa-eye"></i></a>
                        <a href="'.$editUrl.'" class="btn btn-warning btn-sm" title="تعديل"><i class="fas fa-edit"></i></a>
                      </div>';
                })
                ->rawColumns(['active_badge','action'])
                ->make(true);
        }

        // View واحدة فيها الجدول + مودالات
        return view('admin.services.categories');
    }

    /** JSON للتعبئة في مودال العرض/التعديل */
    public function show(ServiceCategory $serviceCategory, Request $request)
    {
        if ($request->ajax()) {
            return response()->json([
                'id'          => $serviceCategory->id,
                'name'        => $serviceCategory->getTranslations('name'),
                'description' => $serviceCategory->getTranslations('description') ?? ['ar'=>null,'en'=>null],
                'active'      => (bool)$serviceCategory->active,
                'created_at'  => optional($serviceCategory->created_at)?->format('Y-m-d H:i'),
                'updated_at'  => optional($serviceCategory->updated_at)?->format('Y-m-d H:i'),
            ]);
        }
        return redirect()->route('service-categories.index');
    }

    /** فتح create التقليدية غير مستخدم (نرجّع للقائمة) */
    public function create() { return redirect()->route('service-categories.index'); }

    /** Store (AJAX) */
    public function store(StoreServiceCategoryRequest $request)
    {
        $row = ServiceCategory::create([
            'name'        => $request->input('name'),
            'description' => $request->input('description', []),
            'active'      => (bool)$request->input('active', true),
        ]);

        if ($request->ajax()) {
            return response()->json(['success'=>true, 'id'=>$row->id, 'message'=>__('تمت الإضافة بنجاح')]);
        }
        return redirect()->route('service-categories.index')->with('success','تمت الإضافة بنجاح');
    }

    /** Edit التقليدية غير مستخدم — للـ AJAX نرجّع JSON */
    public function edit(ServiceCategory $serviceCategory, Request $request)
    {
        if ($request->ajax()) {
            return $this->show($serviceCategory, $request);
        }
        return redirect()->route('service-categories.index');
    }

    /** Update (AJAX) */
    public function update(UpdateServiceCategoryRequest $request, ServiceCategory $serviceCategory)
    {
        $serviceCategory->update([
            'name'        => $request->input('name'),
            'description' => $request->input('description', []),
            'active'      => (bool)$request->input('active', true),
        ]);

        if ($request->ajax()) {
            return response()->json(['success'=>true, 'id'=>$serviceCategory->id, 'message'=>__('تم التحديث بنجاح')]);
        }
        return redirect()->route('service-categories.index')->with('success','تم التحديث بنجاح');
    }
}