<?php

namespace App\Http\Controllers;

use App\Http\Requests\StoreServiceRequest;
use App\Http\Requests\UpdateServiceRequest;
use App\Models\Branch;
use App\Models\Service;
use App\Models\ServiceCategory;
use App\Models\ServiceType;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class ServiceController extends Controller
{
    public function index(Request $request)
    {
        // لفلاتر الـ Blade
        $categories = ServiceCategory::orderBy('id', 'desc')->get();
        $types      = ServiceType::orderBy('id', 'desc')->get();

        if ($request->ajax()) {
            $q = Service::query()
                ->with(['category:id,name', 'type:id,name'])
                ->select([
                    'id',
                    'service_code',
                    'name',
                    'service_type_id',
                    'service_category_id',
                    'price',
                    'duration_value',
                    'duration_unit',
                    'active',
                    'created_at'
                ]);

            // فلاتر
            if ($name = $request->string('name')->toString()) {
                $q->where(function ($qq) use ($name) {
                    $qq->where('name->ar', 'like', "%{$name}%")
                        ->orWhere('name->en', 'like', "%{$name}%");
                });
            }
            if ($typeId = (int) $request->input('type_id')) {
                $q->where('service_type_id', $typeId);
            }
            if ($catId = (int) $request->input('category_id')) {
                $q->where('service_category_id', $catId);
            }
            if ($request->filled('active')) {
                $q->where('active', (int) $request->active === 1);
            }

            return \Yajra\DataTables\Facades\DataTables::of($q)
                ->addIndexColumn()
                ->addColumn('translated_name', fn($r) => $r->getTranslation('name', app()->getLocale()))
                ->addColumn('type_name', fn($r) => optional($r->type)->getTranslation('name', app()->getLocale()))
                ->addColumn('category_name', fn($r) => optional($r->category)->getTranslation('name', app()->getLocale()))
                ->addColumn('price_fmt', fn($r) => number_format((float)$r->price, 2))
                ->addColumn('duration_text', function ($r) {
                    $map = ['days' => 'يوم', 'months' => 'شهر', 'years' => 'سنة'];
                    return $r->duration_value . ' ' . ($map[$r->duration_unit] ?? $r->duration_unit);
                })
                ->addColumn(
                    'active_badge',
                    fn($r) =>
                    $r->active
                        ? '<span class="badge bg-success badge-status">نشط</span>'
                        : '<span class="badge bg-secondary badge-status">غير نشط</span>'
                )
                ->addColumn('action', function ($r) {
                    $show = route('services.show', $r->id);
                    $edit = route('services.edit', $r->id);
                    return '
                      <div class="d-flex justify-content-center gap-1">
                        <a href="' . $show . '" class="btn btn-primary btn-sm" title="عرض"><i class="fas fa-eye"></i></a>
                        <a href="' . $edit . '" class="btn btn-warning btn-sm" title="تعديل"><i class="fas fa-edit"></i></a>
                      </div>';
                })
                ->rawColumns(['active_badge', 'action'])
                ->make(true);
        }

        return view('admin.services.index', compact('categories', 'types'));
    }

    /** شاشة الإضافة */
    public function create()
    {
        return $this->formDependencies(view('admin.services.partials.form'));
    }

    /** حفظ جديد (مع توليد كود الخدمة داخل ترانزاكشن) */
    public function store(StoreServiceRequest $request)
    {
        $data = $this->cleanSettings($request->validated());
        $days = $this->normalizeDays($request->input('days', [])); // خارج الإعدادات

        DB::transaction(function () use ($data, $days, &$service) {

            // كود الخدمة النهائي (آمن)
            $code = $this->generateNextServiceCode();

            $service = Service::create([
                'service_code'        => $code,
                'name'                => $data['name'],
                'service_category_id' => $data['service_category_id'],
                'service_type_id'     => $data['service_type_id'],
                'price'               => $data['price'],
                // 'discount_price'      => $data['discount_price'] ?? null,
                'duration_value'      => $data['duration_value'],
                'duration_unit'       => $data['duration_unit'],
                'start_time'          => $data['start_time'] ?? null,
                'end_time'            => $data['end_time'] ?? null,
                'notes'               => $data['notes'] ?? null,
                'settings'            => $data['settings'] ?? null,
                'days'                => $days,          // JSON
                'active'              => (bool)($data['active'] ?? true),
            ]);

            // علاقات الفروع
            $service->providedBranches()->sync($data['provided_branches'] ?? []);
            $service->soldBranches()->sync($data['sold_branches'] ?? []);
        });

        return redirect()->route('services.index')->with('success', 'تمت إضافة الخدمة بنجاح');
    }

    /** عرض */
    public function show(Service $service)
    {
        $service->load(['category', 'type', 'providedBranches', 'soldBranches']);
        return view('admin.services.show', compact('service'));
    }

    /** شاشة التعديل */
    public function edit(Service $service)
    {
        $service->load(['providedBranches:id', 'soldBranches:id']);
        return $this->formDependencies(
            view('admin.services.partials.form', compact('service')),
            $service
        );
    }

    /** تحديث */
    public function update(UpdateServiceRequest $request, Service $service)
    {
        $data = $this->cleanSettings($request->validated());
        $days = $this->normalizeDays($request->input('days', []));

        $service->update([
            // لا نعدّل service_code في التحديث
            'name'                => $data['name'],
            'service_category_id' => $data['service_category_id'],
            'service_type_id'     => $data['service_type_id'],
            'price'               => $data['price'],
            // 'discount_price'      => $data['discount_price'] ?? null,
            
            'duration_value'      => $data['duration_value'],
            'duration_unit'       => $data['duration_unit'],
            'start_time'          => $data['start_time'] ?? null,
            'end_time'            => $data['end_time'] ?? null,
            'notes'               => $data['notes'] ?? null,
            'settings'            => $data['settings'] ?? null,
            'days'                => $days,
            'active'              => (bool)($data['active'] ?? true),
        ]);

        $service->providedBranches()->sync($data['provided_branches'] ?? []);
        $service->soldBranches()->sync($data['sold_branches'] ?? []);

        return redirect()->route('services.index')->with('success', 'تم تحديث الخدمة بنجاح');
    }

    // =========================================================
    // Helpers
    // =========================================================

    /**
     * تحميل متطلبات الـ form وتمرير suggestedServiceCode في حالة الإضافة.
     */
    private function formDependencies($view, ?Service $service = null)
    {
        $categories = ServiceCategory::orderBy('id', 'desc')->get();
        $types      = ServiceType::orderBy('id', 'desc')->get();
        $branches   = Branch::orderBy('id', 'desc')->get();
        $allServices = Service::orderBy('id', 'desc')->get(['id', 'name']);

        $suggestedServiceCode = $service ? null : $this->predictNextServiceCode();

        return $view->with(compact(
            'categories',
            'types',
            'branches',
            'allServices',
            'suggestedServiceCode'
        ));
    }

    /**
     * تنظيف الإعدادات: إزالة المجموعات غير المفعّلة وتفريغ حقول غير لازمة.
     * (days خارج الإعدادات الآن، فلا نلمسه هنا)
     */
    private function cleanSettings(array $data): array
    {
        $s = $data['settings'] ?? [];

        // pause
        if (empty($s['pause']['enabled'])) unset($s['pause']);

        // transfer
        if (empty($s['transfer']['enabled'])) unset($s['transfer']);

        // renew
        if (empty($s['renew']['enabled'])) unset($s['renew']);

        // refund
        if (empty($s['refund']['enabled'])) {
            unset($s['refund']);
        } else {
            // لو مش exchange احذف الخدمة البديلة
            if (($s['refund']['method'] ?? null) !== 'exchange') {
                unset($s['refund']['exchange_service_id']);
            }
        }

        $data['settings'] = empty($s) ? null : $s;
        return $data;
    }

    /**
     * توحيد days[] كقائمة أكواد أيام صالحة (sat..fri) أو null.
     */
    private function normalizeDays($days)
    {
        if (!is_array($days)) return null;
        $valid = ['sat', 'sun', 'mon', 'tue', 'wed', 'thu', 'fri'];
        $clean = array_values(array_intersect($valid, $days));
        return empty($clean) ? null : $clean; // خزن null لو مفيش اختيار
    }

    /**
     * توقع الكود التالي للعرض في الفورم (بدون Lock).
     * يقرأ أكبر قيمة رقمية من service_code ويزيدها 1؛ يبدأ من 1001 عند الفراغ.
     */
    private function predictNextServiceCode(): string
    {
        $max = DB::table('services')
            ->whereNotNull('service_code')
            ->whereRaw("service_code REGEXP '^[0-9]+$'")
            ->max(DB::raw('CAST(service_code AS UNSIGNED)'));

        $next = $max ? ($max + 1) : 1001;
        return str_pad((string)$next, 4, '0', STR_PAD_LEFT); // 1001, 1002, ...
    }

    /**
     * توليد الكود النهائي داخل ترانزاكشن (مع Lock) لضمان التفرد.
     */
    private function generateNextServiceCode(): string
    {
        $max = DB::table('services')
            ->lockForUpdate()
            ->whereNotNull('service_code')
            ->whereRaw("service_code REGEXP '^[0-9]+$'")
            ->max(DB::raw('CAST(service_code AS UNSIGNED)'));

        $next = $max ? ($max + 1) : 1001;
        return str_pad((string)$next, 4, '0', STR_PAD_LEFT);
    }
}