<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\Offer;
use App\Models\Branch;
use App\Models\Member;
use App\Models\Service;
use App\Models\OfferItem;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use App\Models\PaymentMethod;
use App\Models\ServiceInvoice;
use App\Models\ServiceInvoiceItem;
use Illuminate\Support\Facades\DB;

class ServiceInvoiceController extends Controller
{
    public function index(Request $request)
    {
        // ========== AJAX (DataTables) ==========
        if ($request->ajax()) {

            $invoices = ServiceInvoice::query()
                ->select(['id', 'document_no', 'invoice_no', 'member_id', 'branch_id', 'provider_user_id', 'greg_date', 'total_net'])
                ->with([
                    'member:id,name',
                    'branch:id,name',
                    'provider:id,name',
                ])

                // فلترة الفرع
                ->when($request->branch_id !== null && $request->branch_id !== '', function ($q) use ($request) {
                    $q->where('branch_id', $request->branch_id);
                })

                // فلترة مقدم الخدمة
                ->when($request->provider_user_id !== null && $request->provider_user_id !== '', function ($q) use ($request) {
                    $q->where('provider_user_id', $request->provider_user_id);
                })

                // من تاريخ
                ->when($request->filled('date_from'), function ($q) use ($request) {
                    $q->whereDate('greg_date', '>=', $request->date_from);
                })

                // إلى تاريخ
                ->when($request->filled('date_to'), function ($q) use ($request) {
                    $q->whereDate('greg_date', '<=', $request->date_to);
                })

                // بحث نصّي
                ->when($request->filled('search_text'), function ($q) use ($request) {
                    $s = trim($request->search_text);

                    $q->where(function ($sub) use ($s) {
                        $sub->where('document_no', 'LIKE', "%$s%")
                            ->orWhere('invoice_no', 'LIKE', "%$s%")
                            ->orWhereHas('member', function ($m) use ($s) {
                                $m->where('name', 'LIKE', "%$s%");
                            })
                            ->orWhereHas('branch', function ($b) use ($s) {
                                $b->where('name', 'LIKE', "%$s%");
                            })
                            ->orWhereHas('provider', function ($p) use ($s) {
                                $p->where('name', 'LIKE', "%$s%");
                            });
                    });
                })
                ->orderByDesc('id');

            return datatables()->of($invoices)
                ->addIndexColumn()

                ->addColumn('member_name', fn($r) => $r->member->name ?? '—')
                ->addColumn('branch_name', fn($r) => $r->branch->name ?? '—')
                ->addColumn('provider_name', fn($r) => $r->provider->name ?? '—')

                ->editColumn('greg_date', fn($r) => $r->greg_date ? $r->greg_date->format('Y-m-d') : '—')

                ->editColumn('total_net', fn($r) => number_format($r->total_net, 2))

                ->addColumn('actions', function ($r) {

                    $showUrl  = route('service-invoices.show', $r->id);
                    $editUrl  = route('service-invoices.edit', $r->id);
                    $printUrl = route('service-invoices.print', $r->id);

                    return '
                    <div class="btn-group btn-group-sm" role="group">
                        <a href="' . $showUrl . '" class="btn btn-outline-info" title="عرض">
                            <i class="fas fa-eye"></i>
                        </a>

                        <a href="' . $editUrl . '" class="btn btn-outline-warning" title="تعديل">
                            <i class="fas fa-edit"></i>
                        </a>

                        <a href="' . $printUrl . '" class="btn btn-outline-secondary" title="طباعة" target="_blank">
                            <i class="fas fa-print"></i>
                        </a>
                    </div>';
                })

                ->rawColumns(['actions'])
                ->make(true);
        }

        // ========== إحصائيات ==========
        $stats = [
            'count_all'   => ServiceInvoice::count(),
            'count_today' => ServiceInvoice::whereDate('greg_date', today())->count(),
            'count_month' => ServiceInvoice::whereYear('greg_date', now()->year)
                ->whereMonth('greg_date', now()->month)
                ->count(),
            'sum_net'     => ServiceInvoice::sum('total_net'),
        ];

        $branches  = Branch::select('id', 'name')->get();
        $providers = User::select('id', 'name')->get();

        return view('admin.service_invoices.index', compact('branches', 'providers', 'stats'));
    }


    public function create()
    {
        $branches  = Branch::all();
        $members   = Member::all();
        $providers = User::all(); // أو حسب صلاحياتك
        $services = Service::whereIn('service_type_id', [3, 4])->get();
        $methods  = PaymentMethod::where('active', 1)->get(['id', 'name_ar', 'name_en', 'code']);
        $offers = Offer::whereHas('items.service', function ($q) {
            $q->whereIn('service_type_id', [3, 4]);
        })
            ->with(['items' => function ($q) {
                $q->whereHas('service', function ($qq) {
                    $qq->whereIn('service_type_id', [3, 4]);
                });
            }, 'items.service'])
            ->get();


        $nextDocumentNo = (ServiceInvoice::max('document_no') ?? 0) + 1;

        return view('admin.service_invoices.create', compact(
            'branches',
            'members',
            'providers',
            'services',
            'offers',
            'methods',   // ← مهم جدًا
            'nextDocumentNo'
        ));
    }

    public function store(Request $request)
    {
        $request->validate([
            'branch_id' => [
                'required',
                function ($attribute, $value, $fail) {
                    if ($value != 0 && !\App\Models\Branch::where('id', $value)->exists()) {
                        $fail("The selected branch id is invalid.");
                    }
                }
            ],

            'member_id'          => ['required', 'exists:members,id'],
            'provider_user_id'   => ['required', 'exists:users,id'],
            'greg_date'          => ['required', 'date'],
            'items'              => ['required', 'array', 'min:1'],

            'items.*.type'       => ['required', 'in:service,offer'],
            'items.*.service_id' => ['nullable', 'exists:services,id'],
            'items.*.offer_id'   => ['nullable', 'exists:offers,id'],
            'items.*.offer_service_id' => ['nullable', 'exists:services,id'],

            'items.*.price'      => ['required', 'numeric'],
            'items.*.discount'   => ['nullable', 'numeric'],

            'payments'           => ['required', 'array', 'min:1'],
            'payments.*.method'  => ['required', 'string'],
            'payments.*.amount'  => ['required', 'numeric', 'min:0.01'],
        ]);

        // رقم الوثيقة
        $nextDocumentNo = (ServiceInvoice::max('document_no') ?? 0) + 1;

        // رقم الفاتورة
        // تحديد مصدر كود الفرع
        $rawCode =
            $branch->short_name
            ?? $branch->name_en
            ?? $branch->name
            ?? '';

        // تنظيف الاسم (حروف + أرقام فقط)
        $branchCode = strtoupper(
            preg_replace('/[^A-Za-z0-9]/', '', Str::ascii($rawCode))
        );

        // حماية من القيم الفارغة
        if ($branchCode === '') {
            throw new \Exception('لا يمكن توليد رقم الفاتورة: اسم الفرع غير صالح');
        }

        // رقم الفاتورة
        $invoiceNo =
            $branchCode . '/'
            . now()->format('Y')
            . '/S'
            . str_pad($nextDocumentNo, 6, '0', STR_PAD_LEFT);

        DB::transaction(function () use ($request, $nextDocumentNo, $invoiceNo) {

            $branchId = $request->branch_id == 0 ? null : $request->branch_id;

            // إنشاء الفاتورة
            $invoice = ServiceInvoice::create([
                'document_no'      => $nextDocumentNo,
                'invoice_no'       => $invoiceNo,
                'branch_id'        => $branchId,
                'member_id'        => $request->member_id,
                'provider_user_id' => $request->provider_user_id,
                'created_by'       => auth()->id(),
                'account_user_id'  => auth()->id(),
                'greg_date'        => $request->greg_date,
                'hijri_date'       => $request->hijri_date,
                'notes'            => $request->notes,
            ]);

            // ============================
            //      إجماليات البنود
            // ============================
            $sumPrice = 0;     // إجمالي السعر قبل الخصومات
            $sumDisc  = 0;     // خصومات البنود فقط
            $sumTax   = 0;     // الضريبة قبل الخصم الإضافي (للمقارنة)
            $sumNet   = 0;     // الصافي قبل الخصم الإضافي

            foreach ($request->items as $row) {

                $type = $row["type"];
                $desc = '';
                $serviceId = null;
                $offerId = null;
                $offerServiceId = null;

                // ======= خدمة مباشرة =======
                if ($type === 'service') {

                    $service = Service::findOrFail($row['service_id']);
                    $serviceId = $service->id;

                    // استخراج الاسم الصحيح (JSON دعم)
                    $sName = is_array($service->name)
                        ? ($service->name['ar'] ?? $service->name['en'] ?? reset($service->name))
                        : $service->name;

                    $desc = $sName;
                } else {
                    // ======= عرض =======
                    $offer = Offer::findOrFail($row['offer_id']);
                    $svc   = Service::findOrFail($row['offer_service_id']);

                    $offerId = $offer->id;
                    $offerServiceId = $svc->id;
                    $serviceId = $svc->id;

                    $offerName = is_array($offer->name)
                        ? ($offer->name['ar'] ?? $offer->name['en'] ?? reset($offer->name))
                        : $offer->name;

                    $svcName = is_array($svc->name)
                        ? ($svc->name['ar'] ?? $svc->name['en'] ?? reset($svc->name))
                        : $svc->name;

                    $desc = $offerName . ' - ' . $svcName;
                }

                // ===== الحساب =====
                $qty = 1;
                $price = (float) $row['price'];
                $disc  = (float) ($row['discount'] ?? 0);

                $rowPrice = $qty * $price;

                if ($disc > $rowPrice) $disc = $rowPrice;

                $taxable = $rowPrice - $disc;
                $tax     = $taxable * 0.15;
                $total   = $taxable + $tax;

                // إجماليات البنود
                $sumPrice += $rowPrice;
                $sumDisc  += $disc;
                $sumTax   += $tax;
                $sumNet   += $total;

                // تخزين البند
                ServiceInvoiceItem::create([
                    'service_invoice_id' => $invoice->id,
                    'service_id'         => $serviceId,
                    'offer_id'           => $offerId,
                    'offer_service_id'   => $offerServiceId,
                    'description'        => $desc,
                    'price'              => $price,
                    'discount'           => $disc,
                    'tax'                => $tax,
                    'total'              => $total,
                ]);
            }

            // ============================
            //   الخصم الإضافي + الضريبة بعده
            // ============================
            $extraDiscount = (float) ($request->extra_discount ?? 0);

            // الصافي بعد خصومات البنود
            $afterRowDiscounts = $sumPrice - $sumDisc;

            if ($extraDiscount > $afterRowDiscounts) {
                $extraDiscount = $afterRowDiscounts;
            }

            // الصافي بعد كل الخصومات
            $afterAllDiscounts = $afterRowDiscounts - $extraDiscount;

            // الضريبة النهائية بعد الخصم الإضافي
            $taxAfterExtra = $afterAllDiscounts * 0.15;

            // الصافي النهائي
            $finalNet = $afterAllDiscounts + $taxAfterExtra;

            // ============================
            //        حفظ الفاتورة
            // ============================
            $invoice->update([
                'total_price'      => $sumPrice,
                'total_discount'   => $sumDisc + $extraDiscount,
                'total_tax'        => $taxAfterExtra,       // ← الصحيح
                'total_net'        => $finalNet,            // ← الصحيح
                'payment_methods'  => json_encode($request->payments, JSON_UNESCAPED_UNICODE),
            ]);
        });

        return redirect()->route('service-invoices.index')
            ->with('success', 'تم إنشاء فاتورة الخدمات بنجاح');
    }




    // 🔍 للـ Select2 خدمات
    public function searchServices(Request $request)
    {
        $term = $request->get('term');

        $query = Service::query()->where('active', 1);

        if ($term) {
            $query->where('name', 'like', "%{$term}%");
        }

        return $query->limit(20)->get()->map(function ($s) {
            return [
                'id'    => $s->id,
                'text'  => $s->name,
                'price' => $s->price,
            ];
        });
    }

    // ➕ إضافة عضو سريع من المودال
    public function quickStoreMember(Request $request)
    {
        $data = $request->validate([
            'name' => ['required', 'string', 'max:190'],
            // ضيف باقي الحقول التى تحتاجها للعضو
        ]);

        $member = Member::create($data);

        return response()->json([
            'id'   => $member->id,
            'name' => $member->name,
        ]);
    }


    public function getMembers($branchId)
    {
        $members = Member::query()
            ->when($branchId != 'all', fn($q) => $q->where('branch_id', $branchId))
            ->get(['id', 'name']);

        return response()->json($members);
    }

    public function getOfferServices($offerId)
    {
        $items = OfferItem::where('offer_id', $offerId)
            ->whereHas('service', function ($q) {
                $q->whereIn('service_type_id', [1, 2]);   // ← تصفية الخدمات
            })
            ->with(['service' => function ($q) {
                $q->select('id', 'name', 'price', 'service_type_id');
            }])
            ->get()
            ->map(function ($i) {
                return [
                    'id'    => $i->service_id,
                    'name'  => $i->service_name,     // الاسم محفوظ snapshot
                    'price' => $i->final_price,      // السعر النهائي من العرض
                ];
            });

        return response()->json($items);
    }


    public function show($id)
    {
        $invoice = ServiceInvoice::with(['member', 'provider', 'branch', 'items.service'])
            ->findOrFail($id);

        $paymentMethods = PaymentMethod::where('active', 1)->get()->keyBy('id');

        return view('admin.service_invoices.show', compact('invoice', 'paymentMethods'));
    }

    public function print($id)
    {
        $serviceInvoice = ServiceInvoice::with(['member', 'provider', 'branch', 'items.service'])
            ->findOrFail($id);

        // تحقق من وجود التواريخ قبل التمرير إلى الـ View
        if (!$serviceInvoice->created_at) {
            $serviceInvoice->created_at = now(); // تعيين التاريخ الحالي في حالة عدم وجوده
        }
        if (!$serviceInvoice->service_date) {
            $serviceInvoice->service_date = now(); // تعيين التاريخ الحالي في حالة عدم وجوده
        }
        if (!$serviceInvoice->end_date) {
            $serviceInvoice->end_date = now(); // تعيين التاريخ الحالي في حالة عدم وجوده
        }

        $paymentMethods = PaymentMethod::where('active', 1)
            ->get()
            ->keyBy('id');

        return view('admin.service_invoices.print', compact('serviceInvoice','paymentMethods'));
    }

    public function edit($id)
    {
        $invoice = ServiceInvoice::with('items')->findOrFail($id);

        $sumPrice = $invoice->items->sum(fn($i) => $i->qty * $i->price);
        $sumRowDiscounts = $invoice->items->sum('discount');

        $beforeExtra = $sumPrice - $sumRowDiscounts; // الصافي قبل الضريبة
        $actualNet = $invoice->total_net;

        // معادلة الخصم الإضافي الحقيقية:
        $extraDiscount = ($beforeExtra - ($actualNet / 1.15));

        $extraDiscount = round(max(0, $extraDiscount), 2);

        // ------------------------------

        $branches = Branch::all();
        $members = Member::all();
        $providers = User::all();
        $services = Service::all();
        $offers = Offer::all();

        return view(
            'admin.service_invoices.edit',
            compact('invoice', 'branches', 'members', 'providers', 'services', 'offers', 'extraDiscount')
        );
    }



    public function update(Request $request, $id)
    {
        $request->validate([
            'branch_id' => 'required',
            'member_id' => 'required',
            'provider_user_id' => 'required',
            'greg_date' => 'required|date',
            'items' => 'required|array|min:1',
            'items.*.service_id' => 'required|exists:services,id',
            'items.*.qty' => 'required|integer|min:1',
            'items.*.price' => 'required|numeric|min:0',
            'items.*.discount' => 'nullable|numeric|min:0',
        ]);

        DB::transaction(function () use ($request, $id) {

            $invoice = ServiceInvoice::findOrFail($id);

            $invoice->update([
                'branch_id' => $request->branch_id,
                'member_id' => $request->member_id,
                'provider_user_id' => $request->provider_user_id,
                'greg_date' => $request->greg_date,
                'hijri_date' => $request->hijri_date,
                'notes' => $request->notes,
            ]);

            // حذف البنود القديمة
            ServiceInvoiceItem::where('service_invoice_id', $invoice->id)->delete();

            // إعادة إنشاء البنود
            $totalPrice = $totalDiscount = $totalTax = $totalNet = 0;

            foreach ($request->items as $row) {
                $qty  = (int)$row['qty'];
                $price = (float)$row['price'];
                $disc = (float)$row['discount'];

                $rowPrice = $qty * $price;
                $tax = ($rowPrice - $disc) * 0.15;
                $total = $rowPrice - $disc + $tax;

                ServiceInvoiceItem::create([
                    'service_invoice_id' => $invoice->id,
                    'service_id' => $row['service_id'],
                    'qty' => $qty,
                    'price' => $price,
                    'discount' => $disc,
                    'tax' => $tax,
                    'total' => $total,
                ]);

                $totalPrice    += $rowPrice;
                $totalDiscount += $disc;
                $totalTax      += $tax;
                $totalNet      += $total;
            }

            $invoice->update([
                'total_price'    => $totalPrice,
                'total_discount' => $totalDiscount,
                'total_tax'      => $totalTax,
                'total_net'      => $totalNet,
            ]);
        });

        return redirect()->route('service-invoices.index')
            ->with('success', 'تم تعديل الفاتورة بنجاح');
    }
}