<?php

namespace App\Http\Controllers;

use App\Http\Requests\StoreServiceTypeRequest;
use App\Http\Requests\UpdateServiceTypeRequest;
use App\Models\ServiceType;
use Illuminate\Http\Request;
use Yajra\DataTables\Facades\DataTables;

class ServiceTypeController extends Controller
{
    /**
     * قائمة الأنواع (View + DataTables AJAX)
     */
    public function index(Request $request)
    {
        if ($request->ajax()) {
            $q = ServiceType::query()->select(['id', 'name', 'active', 'created_at']);

            return DataTables::of($q)
                ->addIndexColumn()
                ->addColumn('translated_name', function (ServiceType $r) {
                    return $r->getTranslation('name', app()->getLocale());
                })
                ->addColumn('active_badge', function (ServiceType $r) {
                    return $r->active
                        ? '<span class="badge bg-success">' . __('نشط') . '</span>'
                        : '<span class="badge bg-secondary">' . __('غير نشط') . '</span>';
                })
                ->addColumn('action', function (ServiceType $r) {
                    $showUrl = route('service-types.show', $r->id);
                    $editUrl = route('service-types.edit', $r->id);
                    return '
                        <div class="d-flex justify-content-center align-items-center gap-1">
                          <a href="' . $showUrl . '" class="btn btn-primary btn-sm" title="' . __('عرض') . '">
                            <i class="fas fa-eye"></i>
                          </a>
                          <a href="' . $editUrl . '" class="btn btn-warning btn-sm" title="' . __('تعديل') . '">
                            <i class="fas fa-edit"></i>
                          </a>
                        </div>';
                })
                ->rawColumns(['active_badge', 'action'])
                ->make(true);
        }

        // صفحة واحدة فيها الجدول + مودالات (التي أعطيتُك إياها سابقًا)
        return view('admin.services.types');
    }

    /**
     * إرجاع JSON لتعبئة مودال العرض (والتعديل)
     * - الواجهة (jQuery) بتستدعي /service-types/{id} عبر fetch
     */
    public function show(ServiceType $serviceType, Request $request)
    {
        if ($request->ajax()) {
            return response()->json([
                'id'          => $serviceType->id,
                'name'        => $serviceType->getTranslations('name'),
                'description' => $serviceType->getTranslations('description') ?? ['ar' => null, 'en' => null],
                'active'      => (bool) $serviceType->active,
                'created_at'  => optional($serviceType->created_at)?->format('Y-m-d H:i'),
                'updated_at'  => optional($serviceType->updated_at)?->format('Y-m-d H:i'),
            ]);
        }

        // ما عندنا صفحة عرض منفصلة في نمط "view واحد" => ارجع للقائمة
        return redirect()->route('service-types.index');
    }

    /**
     * فتح صفحة إنشاء تقليدية غير مستخدمة هنا — نرجّع للقائمة (مودال الإضافة بيفتح من نفس الصفحة)
     */
    public function create()
    {
        return redirect()->route('service-types.index');
    }

    /**
     * حفظ جديد (يدعم AJAX)
     */
    public function store(StoreServiceTypeRequest $request)
    {
        $row = ServiceType::create([
            'name'        => $request->input('name'),                 // ['ar'=>..,'en'=>..]
            'description' => $request->input('description', []),      // ['ar'=>..,'en'=>..] أو []
            'active'      => (bool) $request->input('active', true),
        ]);

        if ($request->ajax()) {
            return response()->json([
                'success' => true,
                'id'      => $row->id,
                'message' => __('تمت الإضافة بنجاح'),
            ]);
        }

        return redirect()->route('service-types.index')->with('success', __('تمت الإضافة بنجاح'));
    }

    /**
     * فتح صفحة تعديل تقليدية غير مستخدمة هنا — الواجهة تجيب JSON من show()
     */
    public function edit(ServiceType $serviceType, Request $request)
    {
        // لو الطلب AJAX، رجّع نفس JSON الخاص بالعرض لتعبئة المودال
        if ($request->ajax()) {
            return $this->show($serviceType, $request);
        }

        // وإلا رجّع للقائمة (المودال بيتعامل مع التعديل من نفس الصفحة)
        return redirect()->route('service-types.index');
    }

    /**
     * تحديث (يدعم AJAX)
     */
    public function update(UpdateServiceTypeRequest $request, ServiceType $serviceType)
    {
        $serviceType->update([
            'name'        => $request->input('name'),
            'description' => $request->input('description', []),
            'active'      => (bool) $request->input('active', true),
        ]);

        if ($request->ajax()) {
            return response()->json([
                'success' => true,
                'id'      => $serviceType->id,
                'message' => __('تم التحديث بنجاح'),
            ]);
        }

        return redirect()->route('service-types.index')->with('success', __('تم التحديث بنجاح'));
    }
}