<?php

namespace App\Http\Controllers;

use Carbon\Carbon;
use App\Models\Grade;
use App\Models\Subject;
use App\Models\Classroom;
use Illuminate\Http\Request;
use Yajra\DataTables\DataTables;

class SubjectController extends Controller
{
    public function index(Request $request)
    {
        if ($request->ajax()) {
            $subjects = Subject::select(['subjects.*', 'grades.name as grade_name', 'classrooms.name as class_name'])
                ->leftJoin('grades', 'subjects.grade_id', '=', 'grades.id')
                ->leftJoin('classrooms', 'subjects.classroom_id', '=', 'classrooms.id');

            return DataTables::of($subjects)
                ->filter(function ($query) use ($request) {
                    if ($search = $request->get('search')['value']) {
                        $query->where(function ($q) use ($search) {
                            $q->where('subjects.name->ar', 'LIKE', "%{$search}%")
                                ->orWhere('subjects.name->en', 'LIKE', "%{$search}%")
                                ->orWhere('grades.name->ar', 'LIKE', "%{$search}%")
                                ->orWhere('grades.name->en', 'LIKE', "%{$search}%")
                                ->orWhere('classrooms.name->ar', 'LIKE', "%{$search}%")
                                ->orWhere('classrooms.name->en', 'LIKE', "%{$search}%")
                                ->orWhere('subjects.price', 'LIKE', "%{$search}%");

                        });
                    }
                })
                ->addColumn('translated_name', function ($subject) {
                    return $subject->getTranslation('name', app()->getLocale());
                })
                ->addColumn('price', function ($subject) {
                  return $subject->price;
              })
                ->addColumn('grade_name', function ($subject) {
                    return $subject->grade_name ? json_decode($subject->grade_name, true)[app()->getLocale()] ?? '-' : '-';
                })
                ->addColumn('class_name', function ($subject) {
                    return $subject->class_name ? json_decode($subject->class_name, true)[app()->getLocale()] ?? '-' : '-';
                })
                ->addColumn('action', function ($row) {
                    return '
                    <div class="btn-group text-center">
                        <button class="btn btn-success dropdown-toggle btn-sm btn-pill" type="button" data-bs-toggle="dropdown" aria-expanded="false">
                            <i class="fas fa-cogs"></i> ' .
                        __('dashboard.action') .
                        '
                        </button>
                        <ul class="dropdown-menu text-center">
                            <li>
                                <a class="dropdown-item text-warning edit-btn" href="#"
                                    data-id="' .
                        $row->id .
                        '"
                                    data-name_ar="' .
                        $row->getTranslation('name', 'ar') .
                        '"
                                    data-name_en="' .
                        $row->getTranslation('name', 'en') .
                        '"
                                    data-grade_id="' .
                        $row->grade_id .
                        '"
                                    data-class_id="' .
                        $row->classroom_id .
                        '"
                                  data-price="' .
                        $row->price .
                        '"
                                    data-bs-toggle="modal"
                                    data-bs-target="#editModal">
                                    <i class="fas fa-edit mx-2"></i> ' .
                        __('dashboard.editSub') .
                        '
                                </a>
                            </li>
                           <li>
                    <a class="dropdown-item text-info show-btn" href="' . route('subjects.show', $row->id) . '">
                        <i class="fas fa-eye mx-2"></i> ' . __('dashboard.showSubject') . '
                    </a>
                </li>
                      
                        </ul>
                    </div>
                ';
                })
                ->orderColumn('translated_name', function ($query, $order) {
                    $query->orderByRaw("JSON_UNQUOTE(JSON_EXTRACT(subjects.name, '$.en')) {$order}");
                })
                ->orderColumn('grade_name', function ($query, $order) {
                    $query->orderByRaw("JSON_UNQUOTE(JSON_EXTRACT(grades.name, '$.en')) {$order}");
                })
                ->orderColumn('class_name', function ($query, $order) {
                    $query->orderByRaw("JSON_UNQUOTE(JSON_EXTRACT(classrooms.name, '$.en')) {$order}");
                })
                ->rawColumns(['action'])
                ->make(true);
        }

        $grades = Grade::with('classrooms')->get();
        return view('admin.pages.subject.index', compact('grades'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'grade_id' => 'required|exists:grades,id',
            'classroom_id' => 'required|exists:classrooms,id',
            'name_ar' => 'required|string|max:255',
            'name_en' => 'required|string|max:255',
            'price' => 'required|numeric|min:0',
        ]);

        try {
            // Create new subject with translations
            $subject = Subject::create([
                'grade_id' => $request->grade_id,
                'classroom_id' => $request->classroom_id,
                'price' => $request->price,

                'name' => [
                    'ar' => $request->name_ar,
                    'en' => $request->name_en,
                ],
            ]);
            toastr()->success(__('messages.success'));
            return redirect()->route('subjects.index');
        } catch (\Exception $e) {
            toastr()->success(__('flasher::messages.error'));
            return redirect()->route('subjects.index');
        }
    }

    public function getClasses($grade_id)
    {
        $locale = app()->getLocale(); // Get the current language
        $classes = Classroom::where('grade_id', $grade_id)->get();

        if ($classes->isEmpty()) {
            return response()->json(
                [
                    'message' => __('dashboard.no_classes_found'), // Use a translatable message
                ],
                404,
            );
        }

        $formattedClasses = $classes->map(function ($class) use ($locale) {
            return [
                'id' => $class->id,
                'name' => $class->getTranslation('name', $locale), // Return translated name
            ];
        });

        return response()->json($formattedClasses);
    }

    public function update(Request $request, $id)
    {
      
        $request->validate([
            'name.ar' => 'required|string|max:255',
            'name.en' => 'required|string|max:255',
            'grade_id' => 'required|exists:grades,id',
            'classroom_id' => 'required|exists:classrooms,id',
            'price' => 'required|numeric|min:0',
        ]);

        $subject = Subject::findOrFail($id);
        $subject->update([
            'name' => $request->name,
            'grade_id' => $request->grade_id,
            'classroom_id' => $request->classroom_id,
            'price' => $request->price,

        ]);

        return response()->json(['message' => __('messages.success')]);
    }


    public function sub_update(Request $request, $id)
    {

        $request->validate([
            'name.ar' => 'required|string|max:255',
            'name.en' => 'required|string|max:255',
            'grade_id' => 'required|exists:grades,id',
            'classroom_id' => 'required|exists:classrooms,id',
            'price' => 'required|numeric|min:0',
        ]);

        $subject = Subject::findOrFail($id);
        $subject->update([
            'name' => $request->name,
            'grade_id' => $request->grade_id,
            'classroom_id' => $request->classroom_id,
            'price' => $request->price,

        ]);

        toastr()->success(__('messages.success'));
        return redirect()->back();
    }
    
    public function show($id)
    {
        $subject = Subject::findOrFail($id);
        $grades = Grade::with('classrooms')->get();
        
        $groupedUnits = $subject->semesters->mapWithKeys(function ($semester) {
          return [
              $semester->id => [
                  'name' => $semester->name,   // Semester Name
                  'units' => $semester->units, // Associated Units
              ]
          ];
      });


      $groupedExams = $subject->semesters->flatMap(function ($semester) {
        return $semester->units->flatMap(function ($unit) {
            return $unit->lessons->flatMap(function ($lesson) {
                return collect($lesson->exams)->map(function ($exam) {
                    return [
                        'id' => $exam->id,
                        'name' => $exam->title,
                        'question_groups' => collect($exam->questions)->isNotEmpty()
                            ? collect($exam->questions)->groupBy('group_id')
                            : collect(), // إرجاع مجموعة فارغة إذا لم يكن هناك أسئلة
                    ];
                });
            });
        });
    });

    // dd($groupedExams);
      
        return view('admin.pages.subject.show',compact('subject','grades','groupedUnits','groupedExams'));
    }

    
}
