<?php

namespace App\Http\Controllers;

use Carbon\Carbon;
use App\Models\User;

use App\Models\OfferItem;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use App\Models\PaymentMethod;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Yajra\DataTables\Facades\DataTables;
use App\Models\{Subscription, Service, Offer, Member, Branch};

class SubscriptionController extends Controller
{


    public function index(Request $request)
    {
        // 📌 جزء الـ DataTables (AJAX)
        if ($request->ajax()) {
            $subs = Subscription::with(['member:id,name', 'branch:id,name'])
                ->leftJoin('users as u', 'u.id', '=', 'subscriptions.created_by')
                ->select([
                    'subscriptions.id',
                    'subscriptions.invoice_no',
                    'subscriptions.subscription_type',
                    'subscriptions.item_name',
                    'subscriptions.branch_id',
                    'subscriptions.member_id',
                    'subscriptions.status',
                    'subscriptions.verification_code as verification_code', // 👈 المهم!
                    'u.name as created_by_name'
                ])

                ->orderBy('subscriptions.id', 'DESC');

            // 🔹 فلتر الحالة (من الكروت)
            if ($request->filled('status')) {
                $subs->where('status', $request->status);
            }

            // 🔹 فلتر نوع الاشتراك (service / offer)
            if ($request->filled('type')) {
                $subs->where('subscription_type', $request->type);
            }

            // 🔹 فلتر الفرع
            if ($request->filled('branch_id')) {
                $subs->where('branch_id', $request->branch_id);
            }

            // 🔹 فلتر مسجّل الاشتراك
            if ($request->filled('created_by')) {
                $subs->where('created_by', $request->created_by);
            }

            return datatables()->of($subs)
                ->addIndexColumn()
                ->addColumn('member', fn($r) => $r->member->name ?? '—')
                ->addColumn('branch', fn($r) => $r->branch->name ?? '—')
                ->addColumn('verification_code', function ($r) {
                    if ($r->status !== 'approved') return '—';
                    return '<span class="badge bg-success">' . ($r->verification_code ?? '—') . '</span>';
                })
                ->editColumn('subscription_type', fn($r) => $r->subscription_type === 'service' ? 'خدمة' : 'عرض')
                ->addColumn('status', function ($r) {

                    switch ($r->status) {
                        case 'approved':
                            $badge = '<span class="badge bg-success">
                        <i class="fas fa-check-circle"></i> معتمد
                      </span>';
                            break;

                        case 'under_review':
                            $badge = '<span class="badge bg-warning text-dark">
                        <i class="fas fa-hourglass-half"></i> تحت المراجعة
                      </span>';
                            break;

                        case 'rejected':
                            $badge = '<span class="badge bg-danger">
                        <i class="fas fa-times-circle"></i> مرفوض
                      </span>';
                            break;

                        default:
                            $badge = '<span class="badge bg-secondary">
                        <i class="fas fa-plus-circle"></i> جديد
                      </span>';
                    }

                    $viewBtn = '
                        <a href="' . route('subscriptions.show', $r->id) . '"
                        class="btn btn-sm btn-outline-primary ms-2">
                            <i class="fas fa-eye"></i>
                        </a>
                    ';

                    return '<div class="d-flex justify-content-center align-items-center gap-1">'
                        . $badge .
                        $viewBtn .
                        '</div>';
                })



                ->addColumn('created_by', fn($row) => $row->created_by_name ?? '—')
                ->rawColumns(['created_by', 'status', 'verification_code'])

                ->make(true);
        }

        // 📊 الإحصائيات
        $stats = [
            'new'           => Subscription::where('status', 'new')->count(),
            'under_review'  => Subscription::where('status', 'under_review')->count(),
            'approved'      => Subscription::where('status', 'approved')->count(),
            'rejected'      => Subscription::where('status', 'rejected')->count(),
        ];

        // ✅ البيانات اللازمة للفلاتر
        $allBranches = Branch::select('id', 'name')->get();
        $usersList   = User::select('id', 'name')->get();

        return view('admin.subscriptions.index', compact('stats', 'allBranches', 'usersList'));
    }


    private function generateVerificationNumber()
    {
        // الحصول على آخر رقم تحقق (رقم فقط)
        $last = Subscription::whereNotNull('verification_code')
            ->orderBy('verification_code', 'DESC')
            ->first();

        if (!$last) {
            return '000001';
        }

        // تحويل الرقم لنوع integer
        $num = intval($last->verification_code);

        // زيادة الرقم 1
        $next = $num + 1;

        // تنسيق الرقم ليصبح 6 أرقام
        return str_pad($next, 6, '0', STR_PAD_LEFT);
    }



    public function bulkStatus(Request $request)
    {
        $request->validate([
            'ids' => 'required|array',
            'status' => 'required|in:approved,rejected,under_review',
            'rejection_note' => 'nullable|string|max:500'
        ]);

        $subs = Subscription::whereIn('id', $request->ids)->get();

        foreach ($subs as $sub) {

            if ($request->status === 'approved') {
                $sub->verification_code = $this->generateVerificationNumber();
                $sub->rejection_notes = null;
            } elseif ($request->status === 'rejected') {
                $sub->verification_code = null;
                $sub->rejection_notes = $request->rejection_note;
            } elseif ($request->status === 'under_review') {
                $sub->verification_code = null;
                $sub->rejection_notes = null;
            }

            $sub->status = $request->status;
            $sub->save();
        }

        return response()->json([
            'success' => true,
            'message' => "تم تنفيذ الإجراء بنجاح"
        ]);
    }


    public function printMulti(Request $request)
    {
        // طباعة المحدد
        if ($request->filled('ids')) {
            $ids = explode(',', $request->ids);
            $subs = Subscription::whereIn('id', $ids)->get();
        }
        // طباعة الكل حسب الفلتر
        elseif ($request->filled('all')) {
            $subs = Subscription::when($request->status, function ($q) use ($request) {
                $q->where('status', $request->status);
            })->get();
        } else {
            abort(404);
        }

        return view('admin.subscriptions.print-multi', compact('subs'));
    }



    /** ================================
     * عرض صفحة إنشاء اشتراك جديد
     * ================================ */
    public function create()
    {
        $userBranch = auth()->user()->branch_id
            ? Branch::find(auth()->user()->branch_id)
            : null;

        $branches = $userBranch ? [] : Branch::all();
        // 🟦 1) جلب الخدمات Service فقط (النوع 1 أو 2)
        $services = Service::where('active', 1)
            ->whereIn('service_type_id', [1, 2])
            ->get();


        // 🟧 2) جلب العروض Offer فقط اللي بداخلها خدمات نوعها 1 أو 2
        $offers = Offer::where('active', 1)
            ->whereHas('items.service', function ($q) {
                $q->whereIn('service_type_id', [1, 2]);
            })
            ->with([
                'items' => function ($q) {
                    $q->whereHas('service', function ($qq) {
                        $qq->whereIn('service_type_id', [1, 2]);
                    });
                },
                'items.service'
            ])
            ->get();

        $methods  = PaymentMethod::where('active', 1)->get(['id', 'name_ar', 'name_en', 'code']);

        $nextNo = (Subscription::max('subscription_no') ?? 1000) + 1;

        $users = User::select('id', 'name')->get();


        return view('admin.subscriptions.create', compact(
            'branches',
            'userBranch',
            'services',
            'offers',
            'methods',
            'nextNo',
            'users'
        ));
    }

    /** ================================
     * جلب أعضاء الفرع المحدد
     * ================================ */
    public function getMembers($branchId)
    {
        $members = Member::query()
            ->where('status', '!=', 'inactive')
            ->when($branchId != 'all', fn($q) => $q->where('branch_id', $branchId))
            ->withCount('subscriptions')
            ->get();

        return response()->json($members);
    }




    /** ================================
     * جلب تفاصيل عضو معين
     * ================================ */
    public function getMemberInfo($id)
    {
        $m = Member::where('id', $id)
            ->where('status', '!=', 'inactive')
            ->withCount([
                'subscriptions' => fn($q) => $q->where('status', 'active')
            ])
            ->firstOrFail();

        return response()->json([
            'name' => $m->name,
            'phone' => $m->phone,
            'branch' => $m->branch->name ?? '—',
            'active_subscriptions' => $m->subscriptions_count,
        ]);
    }


    /** ================================
     * جلب الخدمات داخل عرض
     * ================================ */
    public function getOfferServices($offerId)
    {
        $items = OfferItem::where('offer_id', $offerId)
            ->whereHas('service', function ($q) {
                $q->whereIn('service_type_id', [1, 2]);
            })
            ->with([
                'service:id,name,price,duration_value,duration_unit,service_type_id',
                'offer:id,discount_type,discount_value'
            ])
            ->get()
            ->map(function ($i) {

                $service = $i->service;
                $offer   = $i->offer;

                // السعر الأصلي من الخدمة
                $basePrice = (float) ($service->price ?? 0);

                // الخصم من العرض
                $discountType  = $offer->discount_type ?? null;
                $discountValue = (float) ($offer->discount_value ?? 0);

                // حساب الخصم الحقيقي (للعرض فقط)
                $discountAmount = 0;
                if ($discountType === 'percent') {
                    $discountAmount = $basePrice * $discountValue / 100;
                } elseif ($discountType === 'amount') {
                    $discountAmount = $discountValue;
                }

                return [
                    // ===== الأساسيات (كما هي الآن) =====
                    'id' => $i->service_id,
                    'name' => is_array($i->service_name)
                        ? ($i->service_name['ar'] ?? $i->service_name['en'])
                        : $i->service_name,

                    // السعر النهائي (لا نلمسه)
                    'price' => (float) $i->final_price,

                    // ===== البيانات الجديدة (إضافة فقط) =====
                    'base_price' => $basePrice,                 // السعر الأصلي
                    'discount_type' => $discountType,           // percent | amount
                    'discount_value' => $discountValue,         // قيمة الخصم
                    'discount_amount' => round($discountAmount, 2), // الخصم المحسوب

                    // المدة
                    'duration_value' => $i->final_duration_value,
                    'duration_unit'  => $i->final_duration_unit,
                ];
            });

        return response()->json($items);
    }



    /** ================================
     * حفظ الاشتراك
     * ================================ */

    public function store(Request $request)
    {
        $validated = $request->validate([
            'branch_id'             => 'required|integer', // هنسمح بـ 0 = كل الفروع
            'member_id'             => 'required|integer|exists:members,id',
            'subscription_type'     => 'required|in:service,offer',
            'item_id'               => 'required|integer',
            'offer_service_id'      => 'nullable|integer',

            'start_date'            => 'required|date',
            'extra_discount'        => 'nullable|numeric|min:0',
            'service_user_id'       => 'required|integer|exists:users,id',
            'provider_user_id'      => 'required|integer|exists:users,id',
            'created_by'            => 'nullable|integer|exists:users,id',
        ]);

        return DB::transaction(function () use ($validated, $request) {

            $member = Member::findOrFail($validated['member_id']);

            // ============================
            // 1) تحديد الفرع الفعلي
            // ============================
            // لو المستخدم اختار "كل الفروع" (0) → نربط الاشتراك بفرع العضو نفسه
            if ((int)$validated['branch_id'] === 0) {
                if (empty($member->branch_id)) {
                    throw new \Exception('لا يمكن استخدام "كل الفروع" بدون أن يكون للعضو فرع محدد.');
                }
                $branchId = $member->branch_id;
            } else {
                $branchId = $validated['branch_id'];
            }

            $branch = Branch::findOrFail($branchId);

            $type = $validated['subscription_type'];

            // ============================
            // 2) جلب الخدمة / العرض
            // ============================
            if ($type === 'service') {
                $item = Service::findOrFail($validated['item_id']);
            } else {
                $item = Offer::findOrFail($validated['item_id']);
            }

            // ============================
            // 3) حساب السعر والمدة والخصم الأساسي
            // ============================
            $basePrice      = 0;   // السعر قبل أي خصومات
            $baseDiscount   = 0;   // خصم العرض نفسه (لو فيه)
            $durationDays   = 1;   // مدة الاشتراك بالأيام
            $itemName       = is_array($item->name)
                ? ($item->name['ar'] ?? $item->name['en'] ?? '—')
                : ($item->name ?? '—');

            // دالة مساعدة للتحويل لأيام
            $toDays = function ($value, $unit) {
                $v = $value ?? 0;
                $u = strtolower($unit ?? 'days');

                if ($u === 'months') {
                    return $v * 30;
                } elseif ($u === 'years') {
                    return $v * 365;
                }
                return $v; // days
            };

            if ($type === 'service') {
                /** @var Service $service */
                $service = $item;

                $basePrice    = $service->price ?? 0;
                $baseDiscount = 0; // لا يوجد خصم أساسي للخدمة هنا
                $durationDays = $toDays($service->duration_value ?? 1, $service->duration_unit ?? 'days');
            } else {
                /** @var Offer $offer */
                $offer = $item;

                // لو تم اختيار خدمة داخل العرض
                if (!empty($validated['offer_service_id'])) {
                    $offerItem = OfferItem::where('offer_id', $offer->id)
                        ->where('service_id', $validated['offer_service_id'])
                        ->firstOrFail();

                    $basePrice    = $offerItem->price ?? 0;
                    $durationDays = $toDays($offerItem->duration_value ?? 1, $offerItem->duration_unit ?? 'days');
                    $itemName     = $offerItem->name ?? $itemName;
                } else {
                    // احتياطي لو مفيش خدمة محددة
                    $basePrice    = $offer->price ?? 0;
                    $durationDays = $toDays($offer->duration_value ?? 1, $offer->duration_unit ?? 'days');
                }

                // خصم العرض (نفس منطق الـ JS: percent أو amount)
                $discountType  = $offer->discount_type ?? 'amount';
                $discountValue = $offer->discount_value ?? 0;

                if ($discountType === 'percent') {
                    $baseDiscount = ($basePrice * $discountValue) / 100;
                } else {
                    $baseDiscount = $discountValue;
                }
            }

            // ============================
            // 4) تطبيق الخصم الإضافي + الضريبة
            // ============================
            $basePrice        = max(0, $basePrice);
            $extraDiscount    = $validated['extra_discount'] ?? 0;
            $extraDiscount    = max(0, $extraDiscount);

            // إجمالي الخصومات (خصم العرض + الخصم الإضافي)
            $totalDiscount    = $baseDiscount + $extraDiscount;

            // لا تتجاوز الخصم قيمة السعر
            if ($totalDiscount > $basePrice) {
                $totalDiscount = $basePrice;
            }

            // الصافي بعد جميع الخصومات
            $net = max(0, $basePrice - $totalDiscount);

            // الضريبة 15% على الصافي فقط (نفس منطق الواجهة)
            $tax   = $net * 0.15;
            $total = $net + $tax;

            // ============================
            // 5) التواريخ (بداية / نهاية)
            // ============================
            $start = Carbon::parse($validated['start_date']);
            $durationDays = max(1, (int)$durationDays);
            $end   = $start->copy()->addDays($durationDays - 1);

            // ============================
            // 6) توليد رقم الاشتراك والفاتورة
            // ============================
            $lastSubNo = DB::table('subscriptions')
                ->where('branch_id', $branch->id)
                ->lockForUpdate()
                ->max('subscription_no');

            $subscriptionNo = ($lastSubNo ?? 1000) + 1;

            // توليد كود الفرع من الاسم الإنجليزي
            // اختيار المصدر المناسب لكود الفرع
            $rawCode =
                $branch->short_name
                ?? $branch->name_en
                ?? $branch->name
                ?? '';

            // تنظيف الاسم (حروف + أرقام فقط)
            $branchCode = strtoupper(
                preg_replace('/[^A-Za-z0-9]/', '', $rawCode)
            );

            // لو بعد التنظيف فاضي (حالة نادرة جداً)
            if ($branchCode === '') {
                throw new \Exception('لا يمكن توليد كود الفاتورة: اسم الفرع غير صالح');
            }

            $invoiceNo = "{$branchCode}/" . now()->year . "/M{$subscriptionNo}";


            if (Subscription::where('invoice_no', $invoiceNo)->exists()) {
                throw new \Exception("⚠️ رقم الفاتورة مكرر بالفعل: {$invoiceNo}");
            }


            if ($total > 0) {
                // 🔴 اشتراك مدفوع → الدفع إجباري
                $request->validate([
                    'payments'              => 'required|array|min:1',
                    'payments.*.method_id'  => 'required|integer|exists:payment_methods,id',
                    'payments.*.amount'     => 'required|numeric|min:0.01',
                ]);

                // تحقق إن مجموع المدفوع = الإجمالي
                $paid = collect($request->payments)->sum('amount');

                if (abs($paid - $total) > 0.01) {
                    throw new \Exception('⚠️ مجموع المدفوع لا يساوي إجمالي الفاتورة');
                }
            }


            // ============================
            // 7) إنشاء الاشتراك
            // ============================
            $subscription = Subscription::create([
                'subscription_no'   => $subscriptionNo,
                'invoice_no'        => $invoiceNo,
                'branch_id'         => $branch->id,
                'member_id'         => $member->id,
                'subscription_type' => $type,
                'item_id'           => $item->id,
                'item_name'         => $itemName,
                'start_date'        => $start,
                'end_date'          => $end,
                'duration'          => $durationDays,
                'price'             => $basePrice,       // السعر قبل الخصم
                'discount'          => $totalDiscount,   // إجمالي الخصومات (عرض + إضافي)
                'tax'               => $tax,
                'total'             => $total,
                'extra_discount'    => $extraDiscount,   // لو عندك عمود له
                'status'            => 'under_review',
                'active'            => true,
                'service_user_id'   => $validated['service_user_id'],
                'provider_user_id'  => $validated['provider_user_id'],
                'created_by'        => $validated['created_by'] ?? auth()->id(),
                'account_user_id'   => auth()->id(),
            ]);


            if ($total > 0) {
                $subscription->payment_methods = json_encode($request->payments);
                $subscription->save();
            }


            return redirect()
                ->route('subscriptions.invoice', $subscription->id)
                ->with('success', "✅ تم إنشاء الاشتراك بنجاح (رقم الفاتورة: {$invoiceNo}) وهو الآن تحت المراجعة.");
        });
    }

    public function toggle($id)
    {
        $sub = Subscription::findOrFail($id);
        $sub->active = !$sub->active;
        $sub->save();

        // أعد حساب الإحصائيات بعد التغيير
        $stats = [
            'total'     => Subscription::count(),
            'active'    => Subscription::where('active', 1)->count(),
            'inactive'  => Subscription::where('active', 0)->count(),
            'services'  => Subscription::where('subscription_type', 'service')->count(),
            'offers'    => Subscription::where('subscription_type', 'offer')->count(),
        ];

        return response()->json([
            'success' => true,
            'status'  => $sub->active,
            'stats'   => $stats,
        ]);
    }

    public function show($id)
    {
        $sub = Subscription::with(['member', 'branch', 'creator'])->findOrFail($id);

        $paymentMethods = PaymentMethod::where('active', 1)
            ->get()
            ->keyBy('id');

        return view('admin.subscriptions.show', compact('sub', 'paymentMethods'));
    }


    public function invoice(Subscription $sub)
    {
        // علشان الـ Blade يستخدم $sub
        $sub->load(['branch:id,name', 'member:id,name']); // تحسين الأداء
        return view('admin.subscriptions.invoice', compact('sub'));
    }


    public function card($id)
    {
        $sub = Subscription::with(['member', 'branch', 'paymentMethod'])->findOrFail($id);
        return view('admin.subscriptions.card', compact('sub'));
    }
}
