<?php

namespace App\Http\Controllers;

use Validator;
use App\Models\Unit;
use App\Models\Subject;
use Illuminate\Http\Request;

class UnitController extends Controller
{
    public function store(Request $request)
    {
        // Validate the incoming request
        $request->validate([
            'name_ar' => 'required|string|max:255',
            'name_en' => 'required|string|max:255',
            'info_ar' => 'nullable|string',
            'info_en' => 'nullable|string',
            'semester_id' => 'required|exists:semesters,id', // Ensure semester_id exists in the semesters table
        ]);

        // Create the unit with translations and associate with a semester
        $unit = Unit::create([
            'name' => [
                'ar' => $request->name_ar,
                'en' => $request->name_en,
            ],
            'info' => [
                'ar' => $request->info_ar,
                'en' => $request->info_en,
            ],
            'semester_id' => $request->semester_id,
        ]);

        toastr()->success(__('messages.success'));
        return redirect()->back();
    }

    // Update an existing unit
    public function update(Request $request, $id)
    {
        // Validate the incoming request
        $request->validate([
            'name_ar' => 'required|string|max:255',
            'name_en' => 'required|string|max:255',
            'info_ar' => 'nullable|string',
            'info_en' => 'nullable|string',
            'semester_id' => 'required|exists:semesters,id',
        ]);

        // Find the unit by ID
        $unit = Unit::find($id);

        // Update the unit with new translations and semester_id
        $unit->update([
            'name' => [
                'ar' => $request->name_ar,
                'en' => $request->name_en,
            ],
            'info' => [
                'ar' => $request->info_ar,
                'en' => $request->info_en,
            ],
            'semester_id' => $request->semester_id,
        ]);

        toastr()->success(__('messages.success'));
        return redirect()->back();
    }

    public function showUnits($semester_id, $subject_id)
    {
        // Fetch the subject and semester data
        $subject = Subject::findOrFail($subject_id);
        $semester = $subject->semesters->find($semester_id);  // Get the semester by ID

        // Fetch all units for the specific semester
        $units = $semester->units;

        // Return the view with the units
        return view('admin.pages.subject.unit', compact('units', 'subject', 'semester'));
    }


    public function getUnitsBySemester($semesterId)
    {
      
        $locale = app()->getLocale(); // Get the current language
        $units = Unit::where('semester_id', $semesterId)->get();

      

        $unitsAll = $units->map(function ($unit) use ($locale) {
            return [
                'id' => $unit->id,
                'name' => $unit->getTranslation('name', $locale), // Return translated name
            ];
        });

        return response()->json($unitsAll);
    }
}
