<?php

namespace App\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\Validator;

class UpdateBranchRequest extends FormRequest
{
    public function authorize(): bool
    {
        // عدّلها إن كنت تستخدم Policies/Permissions
        return true;
    }

    /**
     * تجهيز القيم قبل التحقق
     */
    protected function prepareForValidation(): void
    {
        $this->merge([
            'phones'  => $this->input('phones', []),
            'address' => $this->input('address', []),
            'active'  => filter_var(
                $this->input('active', true),
                FILTER_VALIDATE_BOOLEAN,
                FILTER_NULL_ON_FAILURE
            ) ?? false,
        ]);
    }

    public function rules(): array
    {
        return [
            // الاسم المترجم (Spatie Translatable)
            'name.ar' => ['required', 'string', 'max:255'],
            'name.en' => ['required', 'string', 'max:255'],

 

            // ✅ الجديد
            'short_name' => ['required', 'string', 'max:50'],
            'gender'     => ['required', 'in:0,1'],

            // العلاقات
            'branch_category_id' => ['required', 'exists:branch_categories,id'],
            'branch_type_id'     => ['required', 'exists:branch_types,id'],
            'city_id'            => ['required', 'exists:states,id'],

            // الهاتف الأساسي
            'primary_phone'      => ['required', 'string', 'max:30', 'regex:/^[0-9+\-\s()]+$/'],

            // هواتف فرعية (JSON)
            'phones'             => ['nullable', 'array', 'max:20'],
            'phones.*.label.ar'  => ['required_with:phones.*.number', 'string', 'max:100'],
            'phones.*.label.en'  => ['required_with:phones.*.number', 'string', 'max:100'],
            'phones.*.number'    => ['required_with:phones.*.label.ar', 'string', 'max:30', 'regex:/^[0-9+\-\s()]+$/'],

            // حالة وحقول إضافية
            'active'               => ['required', 'boolean'],
            'min_age'              => ['nullable', 'integer', 'min:0', 'max:120'],
            'max_discount_percent' => ['nullable', 'numeric', 'min:0', 'max:100'],

            // العنوان (JSON اختياري)
            'address'              => ['nullable', 'array'],
            // أمثلة تفصيلية (اختيارية):
            // 'address.full.ar'     => ['nullable', 'string', 'max:255'],
            // 'address.full.en'     => ['nullable', 'string', 'max:255'],
            // 'address.street.ar'   => ['nullable', 'string', 'max:255'],
            // 'address.street.en'   => ['nullable', 'string', 'max:255'],
            // 'address.district.ar' => ['nullable', 'string', 'max:255'],
            // 'address.district.en' => ['nullable', 'string', 'max:255'],
            // 'address.building_no' => ['nullable', 'string', 'max:50'],
            // 'address.postal_code' => ['nullable', 'string', 'max:20'],
            // 'address.tax_no'      => ['nullable', 'string', 'max:50'],

            // الأهداف الشهرية (اختيارية ضمن نفس الطلب)
            'goals.year'             => ['nullable', 'integer', 'min:2000', 'max:2100'],
            'goals.items'            => ['nullable', 'array', 'size:12'], // 12 شهر
            'goals.items.*.month'    => ['required_with:goals.items', 'integer', 'between:1,12'],
            'goals.items.*.actual'   => ['nullable', 'numeric', 'min:0'],
            'goals.items.*.target'   => ['nullable', 'numeric', 'min:0'],
        ];
    }

    /**
     * تحقق إضافي: لو اتبعت الأهداف، تأكد أن الأشهر بالضبط 1..12 بدون تكرار
     */
    public function withValidator(Validator $validator): void
    {
        $validator->after(function (Validator $v) {
            $items = $this->input('goals.items', []);
            if (!$items) {
                return; // لم تُرسل أهداف؛ لا داعي لفحص إضافي
            }

            $months = array_map(
                fn($row) => (int) ($row['month'] ?? 0),
                $items
            );
            sort($months);

            if ($months !== range(1, 12)) {
                $v->errors()->add('goals.items', 'يجب إدخال صف لكل شهر من 1 إلى 12 بدون تكرار.');
            }
        });
    }

    public function messages(): array
    {
        return [
            'primary_phone.regex'   => 'صيغة رقم الهاتف غير صحيحة.',
            'phones.*.number.regex' => 'صيغة رقم الهاتف غير صحيحة.',
            'goals.items.size'      => 'يجب إدخال 12 صفًا (شهرًا) للأهداف الشهرية.',
        ];
    }

    public function attributes(): array
    {
        return [
            'name.ar'                => 'الاسم (ar)',
            'name.en'                => 'الاسم (en)',
            'branch_category_id'     => 'فئة الفرع',
            'branch_type_id'         => 'نوع الفرع',
            'city_id'                => 'المدينة',
            'primary_phone'          => 'الهاتف الأساسي',
            'phones.*.label.ar'      => 'مسمّى الهاتف (ar)',
            'phones.*.label.en'      => 'Label (en)',
            'phones.*.number'        => 'رقم الهاتف',
            'active'                 => 'الحالة',
            'min_age'                => 'الحد الأدنى للعمر',
            'max_discount_percent'   => 'الحد الأقصى للخصم بالنسبة المئوية',
            'goals.year'             => 'السنة',
            'goals.items'            => 'الأهداف الشهرية',
            'goals.items.*.month'    => 'الشهر',
            'goals.items.*.actual'   => 'المبيعات الفعلية',
            'goals.items.*.target'   => 'المبيعات المستهدفة',
        ];
    }
}